function recalculate_view(view) {
    if (view == "" || view == null || view == undefined) {
        Window.Warning("Error", "No view variable found");
        return;
    }
    let templ = Template.GetCurrent();

    /* Create variable with view string for D3PLOT item "#VTC Recalculate" to pick up */
    new Variable(templ, "VIEW", "View to Recalculate", view, "string");

    /* Search for image file items starting with specific name */
    let image_item_str = `automotive_assessment_image_${view}`;

    /* We will populate an array of items matching the specific name */
    let image_items = [];

    /* Loop through all pages to find relevant image file items */
    let pages = templ.GetAllPages();
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, image_item_str.length) == image_item_str) {
                /* Check that any additional characters are numbers and not a different view. For
                 * example, if the view is `front`, we want to find:
                 * automotive_assessment_image_front
                 * automotive_assessment_image_front_1
                 * automotive_assessment_image_front_2
                 *
                 * but not:
                 * automotive_assessment_image_front_centre
                 * automotive_assessment_image_front_centre_1
                 */
                let suffix = item.name.substring(image_item_str.length);
                if (suffix.length == 0 || (suffix.length > 1 && suffix[0] == "_" && suffix[1].match(/\d/) != null)) {
                    /* Report any items erroneously given the identifier name */
                    if (item.type != Item.IMAGE_FILE) {
                        LogError(`Item "${item.name}" on page ${p + 1} is not an Image File item.`);
                        continue;
                    }
                    LogPrint(`Found item "${item.name}" on page ${p + 1} with filename: ${item.file}`);
                    image_items.push(item);
                }
            }
        }
    }
    if (image_items.length == 0) {
        Window.Warning("Error", "Could not find Image File items to update.");
        return;
    }

    /* Next, we need to "unlock" the MP4 file associated with the view we wish to edit.
     * We do this by switching the image file items to point to a different file. Once
     * we have run the D3PLOT item "#VTC Recalculate", we will restore the original filename. */
    let template_dir = templ.GetVariableValue("TEMPLATE_DIR");

    let missing_data_mp4 = "MissingData.mp4";

    /** if the template filename contains "_CN.or" then set the language to Chinese */
    if (/_CN\.or/i.test(templ.filename)) {
        missing_data_mp4 = "MissingData_CN.mp4";
    }

    for (let item of image_items) {
        item.file = template_dir + `/../../scripts/vtc_videos/images/${missing_data_mp4}`;
        generate_specific_item(item);
    }

    /* Next, we move the original video to a temporary location so that D3PLOT can create a new
     * one. */
    let video_filename = templ.GetVariableValue("VIDEO_PATH") + view + ".mp4";
    let video_tempname = video_filename + ".temp";
    if (File.Exists(video_filename)) {
        let success = File.Copy(video_filename, video_tempname);
        if (!success) {
            Window.Warning(
                "Error",
                `Unable to copy video to temporary file.\nOriginal file: ${video_filename}\nDestination: ${video_tempname}`
            );
            return;
        }
        success = File.Delete(video_filename);
        if (!success) {
            Window.Warning("Error", `Unable to delete video file: ${video_filename}`);
            return;
        }
    }

    /* Generate the D3PLOT item to allow the user to edit the view. */
    generate_item(`#VTC Recalculate View`);

    /* Now we return the image file items to their original state. */
    if (!File.Exists(video_filename)) {
        /* If we can't find a new video, we try to restore the original video from the
         * temporary location. */
        if (File.Exists(video_tempname)) {
            let success = File.Copy(video_tempname, video_filename);
            if (!success) {
                Window.Warning(
                    "Edit View",
                    `New video not found at: ${video_filename}\nUnable to copy video from temporary file to original location.\nTemporary file: ${video_tempname}\nDestination: ${video_filename}`
                );
                return;
            }
            success = File.Delete(video_tempname);
            if (!success) {
                LogWarning(`Unable to delete temporary video file: ${video_tempname}`);
            }
            for (let item of image_items) {
                item.file = video_filename;
                generate_specific_item(item);
            }
            Window.Warning("Edit View", `New video not found at: ${video_filename}\nOriginal video restored.`);
        } else {
            Window.Warning(
                "Edit View",
                `New video not found at: ${video_filename}\nUnable to restore original video from: ${video_tempname}\nClick "Edit View" to try again.`
            );
        }
    } else {
        /* If the new video exists, we simply update the filename and regenerate. */
        for (let item of image_items) {
            item.file = video_filename;
            generate_specific_item(item);
        }
        File.Delete(video_tempname);
        Window.Information("Edit View", "Video updated successfully.");
    }

    let sizeVariable = Variable.GetFromName(templ, view + "_size");
    if (sizeVariable != null) {
        let fileSize = File.Size(video_filename);
        fileSize = fileSize / 1000000;
        sizeVariable.value = fileSize.toFixed(1) + "MB";
    }
}

/**
 * Wrapper function to generate a specific item and return its <active> property to its original
 * value.
 * @param {Item} item
 */
function generate_specific_item(item) {
    let active = item.active;
    item.active = true;
    item.Generate();
    item.active = active;
}
