/** set up global template variable for the current template */
var templ = Template.GetCurrent();

/* Utils.Version was added for REPORTER in Version 21. So based on if this function is present or not we can check which version we are running */
if (typeof Utils.Version != "function") {
    let message = `This workflow is only available for Oasys REPORTER version 21 and above.`;
    if (Batch()) {
        LogError(message);
    } else {
        Window.Message("Workflow not available", message);
    }
    Exit();
}
/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
/** set the PASS/FAIL text on the results table on individual page */
var status = { success: true, missing: [], invalid: [] };
try {
    on_load_script();
} catch (error) {
    finish_script(false, error.message);
}

/**
 * Remove temporary variables and set required variables to default values
 */
function initialise_variables() {
    create_variable_if_it_does_not_exist(`LSDYNA_RESULT_FILES`);
    let JOB_CONTROL = new Variable(templ, `JOB_CONTROL`, "", "Run");
    let vcI_rep_var = new Variable(templ, `VALIDATION_RESULT`, "Default", "-");
    create_variable_if_it_does_not_exist(
        `SIMVT_SKIP_GENERATE`,
        `Skip generating the template if another script will already generate`,
        "no"
    );
    /** set the default values for the variables expected by '#SIMVT T/HIS check and do assessment'
     *  note that these variables may be defined via the batch command-line */
    create_variable_if_it_does_not_exist(`SIMVT_SETTINGS_FILE`);
    create_variable_if_it_does_not_exist(`SIMULATION_DATA_PATH`);
    create_variable_if_it_does_not_exist(`TEST_DATA_PATH`);
    create_variable_if_it_does_not_exist(`OUTPUT_DIR`);

    /** these variables are only needed for the Euro NCAP Far Side template */
    create_variable_if_it_does_not_exist(`SIM_MAX_HE`, "Simulation max head excursion displacement", "Missing");
    create_variable_if_it_does_not_exist(`SIM_T_MAX_HE`, "Simulation max head excursion time", "Missing");
    create_variable_if_it_does_not_exist(`SIM_T_END`, "Simulation end time", "Missing");
    create_variable_if_it_does_not_exist(
        `SIM_STATUS`,
        "Missing/PASS/FAIL simulation duration from 1.2*SIM_T_MAX_HE",
        "Missing"
    );
}

function create_variable_if_it_does_not_exist(name, description = "", value = "", type = "General") {
    let new_var = Variable.GetFromName(templ, name);
    if (!new_var) {
        new_var = new Variable(templ, name, description, value, type, false, true);
    }
}

/** generate the REPORTER item with helper functions so that functions in:
 *  %WORKFLOWS_DIR%\scripts\sim_vs_test_correlation\post\reporter\reporter_helper_functions.js
 * can be used in this script (like importing a module). Note we need to do this after set_workflows_dir_variable()
 * so that %WORKFLOWS_DIR% is defined*/
function load_helper_functions() {
    generate_item(`helper_functions`);
}

/** generate the REPORTER item with the function to parse the reporter json */
function load_parse_reporter_json_function() {
    generate_item(`parse_reporter_json`);
}

function load_variables_from_command_line() {
    /** defined error messages array */
    let argument_info = { error_messages: [] };
    /** first check all the possible batch arguments for validity */
    check_batch_arguments(argument_info, `OUTPUT_DIR`, true);
    let simvt_settings_file_defined = check_batch_arguments(argument_info, `SIMVT_SETTINGS_FILE`);
    /** list the simulation variable variants */
    let simulation_var_names = [`SIMULATION_DATA_PATH`, `RESULTS_FILE_THIS`];
    let sim_path_defined = check_batch_arguments(argument_info, simulation_var_names);
    /** list the test variable variants */
    let test_var_names = [`TEST_DATA_PATH`, `TEST_FILE`];
    let test_path_defined = check_batch_arguments(argument_info, test_var_names);

    /** then check for a valid combination */

    /** if output dir is invalid then we cannot proceed */
    if (!argument_info[`OUTPUT_DIR`]) {
        throw Error(argument_info.error_messages.join(`\n`));
    }
    if (simvt_settings_file_defined || (sim_path_defined && test_path_defined)) {
        /** valid combination found so set the variables accordingly */
        LogPrint(`Valid combination of command line arguments for SimVT to run`);
    } else {
        /** if we have no settings file then both SIMULATION_DATA_PATH and TEST_DATA_PATH must be defined else we cannot proceed */
        LogError(`Invalid/incomplete combination of command line arguments for SimVT to run`);
        throw Error(argument_info.error_messages.join(`\n`));
    }
}

function on_load_script() {
    /** initialise the variables */
    initialise_variables();
    /** set the workflows directory variable if it is not defined - it should be defined if the template has been saved as an orr
     * this must be run before loading the helper functions */
    // set_workflows_dir_variable();

    /** load the helper functions */
    load_helper_functions();

    /** load the parse reporter json function */
    load_parse_reporter_json_function();

    /** if running in batch, load the variables from the command line arguments */
    if (Batch()) load_variables_from_command_line();

    /** Clear summary tables */
    // LogWarning(`pre - clean_up_table`);
    // clean_up_table();

    /* Create a fresh, empty AAW_job_control_variables.csv in case it has data from previous runs */
    initialise_job_control_variables_csv();

    /* To begin with, deactivate all #SIMVT items as we want them to only be generated when we explicitly call generate... */
    activate_all_simvt_items(false);

    /** The SimVT script runs T/HIS first and if that was successful then we finish script */
    generate_item(`#SIMVT T/HIS check and do assessment`);

    /** check the status of the JOB after generating the T/HIS item */
    check_job_status();

    /** try and parse the reporter JSON */
    parse_reporter_json();

    finish_script(true);
}

/**
 * checks the values of JOB_CONTROL and JOB_MESSAGE and throws and error if their status represents an error
 */
function check_job_status() {
    /** update the JOB control status */
    generate_item(`#SIMVT REPORTER read job control`);

    /** parse the job control variables to determine if T/HIS was successful */
    let job_control = Variable.GetFromName(Template.GetCurrent(), "JOB_CONTROL");
    let job_message_var = Variable.GetFromName(Template.GetCurrent(), "JOB_MESSAGE");

    if (!job_control) throw Error(`%JOB_CONTROL% is not defined so status of T/HIS assessment is unknown.`);

    let msg = `%JOB_CONTROL% is ${job_control.value}`;
    if (job_message_var && job_message_var.value != "") msg = job_message_var.value;

    if (job_control.value != "Skip") throw Error(msg);
}

/* Delete AAW_job_control_variables.csv in case it has data from previous runs */
function initialise_job_control_variables_csv() {
    let templ = Template.GetCurrent();
    let reporter_temp = get_expanded_variable_value(templ, `REPORTER_TEMP`);
    if (reporter_temp == null || (reporter_temp != null && !File.IsDirectory(reporter_temp))) {
        finish_script(false, `Could not find REPORTER_TEMP directory.`);
    }
    let f_csv_name = `${reporter_temp}/AAW_job_control_variables.csv`;
    if (File.Exists(f_csv_name)) {
        let success = File.Delete(f_csv_name);
        if (!success) {
            finish_script(false, `Unable to delete existing variables file: ${f_csv_name}`);
        }
    }
    let f_reporter_data_name = `${reporter_temp}/reporter_data.json`;
    if (File.Exists(f_reporter_data_name)) {
        let success = File.Delete(f_reporter_data_name);
        if (!success) {
            finish_script(false, `Unable to delete existing variables file: ${f_reporter_data_name}`);
        }
    }
    /* Replace it with an empty file so that `#SIMVT REPORTER read variables from PRIMER job control`
     * item always has something to read. */
    let f_csv = new File(f_csv_name, File.WRITE);
    f_csv.Close();
}

/**
 * @returns {Boolean}
 * @example
 * let success = get_output_dir();
 */
function get_output_dir() {
    let templ = Template.GetCurrent();

    let output_dir;
    if (Batch()) {
        /* First check to see if we can find a settings file.
         * We check SIMVT_SETTINGS_FILE first
         */
        output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);

        let valid_output_dir = false;
        if (output_dir != null && File.IsDirectory(output_dir)) {
            LogError(`${output_dir} is not a directory`);
            valid_output_dir = true;
        } else {
            if (output_dir == null) {
                LogPrint(`REPORTER Variable %OUTPUT_DIR% was not defined.`);
            } else {
                LogPrint(
                    `Settings file specified by REPORTER Variable %SIMVT_SETTINGS_FILE% could not be found: ${output_dir}`
                );
            }
        }

        if (!valid_output_dir) {
            return false;
        }
    } else {
        let ans = Window.Question(
            `Select output directory file`,
            `Select the directory where you want to save the report and outputs`,
            Window.OK | Window.CANCEL
        );
        if (ans == Window.CANCEL) {
            return false;
        }

        output_dir = Window.GetDirectory();

        if (!output_dir) {
            /* User clicked cancel */
            return false;
        }

        if (!File.IsDirectory(output_dir)) {
            LogError(`${output_dir} is not a directory`);
            return false;
        }

        set_output_file_variable(output_dir);

        return true;
    }
}

/**
 * Sets the REPORTER variables %OUTPUT_DIR%
 * The latter 2 are extracted from the settings file
 * specified keyword file.
 * @param {string} output_dir Absolute path and filename of settings file
 */
function set_output_file_variable(output_dir) {
    let templ = Template.GetCurrent();

    /* If output_dir is relative path, make it absolute path. We need to do this so that
     * any other variables that end up being derived from it (DEFAULT_DIR, RESULTS_DIR, OUTPUT_DIR,
     * etc.) are absolute paths too. When REPORTER generates an Oasys item, it sets -start_in to
     * the job file directory, so it gets confused if the relative paths we have defined are
     * relative to a different starting directory. Converting everything to absolute paths avoids
     * this problem. */
    output_dir = convert_to_absolute_path(output_dir, `%OUTPUT_DIR%`);

    let path_index = Math.max(output_dir.lastIndexOf("/"), output_dir.lastIndexOf("\\"));
    let filename = output_dir.substring(path_index + 1);
    let default_job = filename; //the name of the REPORT will be the same as the output folder by default

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * OUTPUT_DIR should be temporary; DEFAULT_DIR and DEFAULT_JOB are not temporary. */
    let output_dir_var = new Variable(templ, `OUTPUT_DIR`, `Output Directory`, output_dir, `Directory`, false, true);
    let default_dir_var = new Variable(
        templ,
        `DEFAULT_DIR`,
        `Reporter default directory`,
        output_dir,
        `Directory`,
        false,
        false
    );
    let default_job_var = new Variable(
        templ,
        `DEFAULT_JOB`,
        `Reporter default jobname`,
        default_job,
        `File(basename)`,
        false,
        false
    );
}

/**
 * Sets the REPORTER %WORKFLOW_DIR% variable - if it is already defined then use the existing value, provided it is valid.
 * Otherwise use the "%TEMPLATE_DIR%/../../" and if that is also not valid then prompt the user to select it until they selecct
 * a valid workflows dir.
 */
function set_workflows_dir_variable() {
    let templ = Template.GetCurrent();
    let workflows_dir;

    let workflows_dir_var = Variable.GetFromName(templ, "WORKFLOWS_DIR");
    if (workflows_dir_var && check_or_get_valid_workflows_dir(workflows_dir_var.value)) {
        return;
    }

    /**use the template dir if WORKFLOWS_DIR is not defined or valid */

    let template_dir = Variable.GetFromName(templ, "TEMPLATE_DIR").value;

    /** workflows directory is "%TEMPLATE_DIR%/../../" so get parent twice */
    workflows_dir = get_parent(get_parent(template_dir));

    /** get valid directory - keep showing window until user selects a valid workflows dir */
    workflows_dir = check_or_get_valid_workflows_dir(workflows_dir, true);

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * WORKFLOWS_DIR should not be temporary*/
    workflows_dir_var = new Variable(
        templ,
        `WORKFLOWS_DIR`,
        `Workflows Directory`,
        workflows_dir,
        `Directory`,
        false,
        false
    );
}

/**
 * this checks if the path is a valid workflow directory and if it is not it gets user to select a directory untill they select a valid one
 * @param {String} workflows_dir path
 * @param {Boolean} [selection_window = false] if true the user is prompted to select a workflow directory recurrsively until they get a valid one
 * @returns {?String} workflows_dir
 */
function check_or_get_valid_workflows_dir(workflows_dir, selection_window = false) {
    let msg;
    let expected_file_in_workflow_dir = `${workflows_dir}/scripts/automotive_assessments/modules/post/this/proxy.mjs`;

    if (!workflows_dir) {
        msg = `<workflows_dir> is undefined`;
    } else if (!File.IsDirectory(workflows_dir)) {
        msg = `"${workflows_dir}" does not exist`;
    } else if (!File.Exists(expected_file_in_workflow_dir)) {
        msg = `"${expected_file_in_workflow_dir}" does not exist so workflows_dir is invalid`;
    } else {
        return workflows_dir;
    }

    if (selection_window) {
        Window.Warning(
            `Select Workflows Directory`,
            `${msg}\nPlease select the workflows directory to use.`,
            Window.OK
        );
        workflows_dir = Window.GetDirectory();

        return check_or_get_valid_workflows_dir(workflows_dir, selection_window);
    }

    LogWarning(msg);
    return null;
}

/**
 * Common function for finishing the script.
 * Complete the generation of the template if we have all the required information.
 * @param {Boolean} can_generate Whether or not to generate the template
 * @param {string} [msg] Error message (provide when not generating)
 * @example
 * finish_script(Template.GetCurrent(), false, `Keyword file not provided`);
 */
function finish_script(can_generate, msg) {
    let templ = Template.GetCurrent();
    /* Complete the template generation if we have the required information,
     * otherwise end with message */

    // if the "#AAW SIMVT REPORTER on load script" item exists then we skip the global generation
    // as it the #AAW REPORTER on load script is already doing the generation.
    let AAW_SIMVT_item = get_item(`#AAW SIMVT REPORTER on load script`, false);

    if (can_generate) {
        if (!AAW_SIMVT_item) {
            /* Deactivate this script item to avoid entering an infinite loop */
            activate_master_page_item(`#SIMVT REPORTER on load script`, false);
            if (templ.GetVariableValue(`SIMVT_SKIP_GENERATE`) == `no`) {
                templ.Generate();
            }
        } else {
            activate_master_page_item(`#AAW SIMVT REPORTER on load script`, true);
        }
    } else {
        if (Batch()) LogError(msg);
        else {
            Window.Message(`Report automation`, msg);
        }
    }

    /* Whether or not we generated, deactivate all the script items except this script item. */
    activate_all_simvt_items(false);
    if (!AAW_SIMVT_item) {
        activate_master_page_item(`#SIMVT REPORTER on load script`, true);
        /** change the template view to presentation mode and update before exiting the script*/
        templ.view = Reporter.VIEW_PRESENTATION;
        templ.Update();
        Exit();
    }
}

/**
 * Activates or deactivates #activate_normal_page_simvt_items items on all pages (master page
 * and normal pages) so they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_all_simvt_items(false);
 */
function activate_all_simvt_items(active) {
    activate_master_page_simvt_items(active);
    activate_normal_page_simvt_items(active);
}

/**
 * Activates or deactivates #SIMVT items on normal pages so that
 * they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_normal_page_simvt_items(false);
 */
function activate_normal_page_simvt_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let hash_item_str = `#SIMVT`;

    /* Search for relevant items on other pages */
    let pages = templ.GetAllPages();
    for (let page of pages) {
        let items = page.GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, hash_item_str.length) == hash_item_str) {
                /* (De)activate all #SIMVT items */
                item.active = active;
            }
        }
    }
}

/**
 * Activates or deactivates #SIMVT master page items so that they
 * are only generated once (they sit on the Master page so would be generated for every page otherwise).
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_master_page_simvt_items(false);
 */
function activate_master_page_simvt_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let hash_item_str = `#SIMVT`;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let items = master.GetAllItems();

    for (let item of items) {
        if (item.name.substring(0, hash_item_str.length) == hash_item_str) {
            /* (De)activate all #SIMVT items */
            item.active = active;
        }
    }
}

/**
 * Activates or deactivates a specific item on the Master page so that it is only generated when
 * required.
 * @param {string} item_name The item name (the first item matching this name will be actioned)
 * @param {boolean} active Whether to activate or deactivate the item
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * activate_master_page_item(`#SIMVT REPORTER on load script`, false);
 */
function activate_master_page_item(item_name, active, expected = true) {
    let templ = Template.GetCurrent();

    /* Syntax for log messages */
    let de = `de`;
    if (active) de = ``;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let match = false;
    let items = master.GetAllItems();
    for (let item of items) {
        /* Deactivate this script item */
        if (item.name.substring(0, item_name.length) == item_name) {
            item.active = active;
            match = true;
            LogPrint(`Successfully ${de}activated master page item "${item_name}".`);
            break;
        }
    }

    if (!match && expected) {
        LogError(`Could not find master page item "${item_name}" to ${de}activate it.`);
        Exit();
    }
}

/**
 * Finds the first item in the template with the specified <item_name> and generates it.
 * Looks at the master page first, then all the normal pages.
 * @param {string} item_name The name of the item to be generated
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * generate_item(`#SIMVT REPORTER read variables from PRIMER job control`);
 */
function generate_item(item_name, expected = true) {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    pages.unshift(template.GetMaster());
    let match = false;
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name == item_name) {
                LogPrint(`Found item "${item_name}" on page ${p + 1}. Generating...`);
                match = true;
                /* Activate the item before generating it, then return it to its original status */
                let active = item.active;
                item.active = true;
                item.Generate();
                item.active = active;
                break;
            }
        }
        if (match) break;
    }
    if (!match && expected) {
        LogError(`Could not find item "${item_name}" in order to generate it.`);
        Exit();
    }
}

/**
 * Finds the first item in the template with the specified <item_name> and returns it.
 * Looks at the master page first, then all the normal pages.
 * @param {string} item_name The name of the item to be generated
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @returns {?Item}
 * @example
 * let item = get_item(`#SIMVT REPORTER read variables from PRIMER job control`);
 */
function get_item(item_name, expected = true) {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    pages.unshift(template.GetMaster());
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name == item_name) {
                LogPrint(`Found item "${item_name}" on page ${p + 1}: "${pages[p].name}". `);
                return item;
            }
        }
    }
    if (expected) {
        LogError(`Could not find item "${item_name}" in order to generate it.`);
        Exit();
    }
    return null;
}

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}

/**
 * Searches a directory for filenames matching either Arup or LSTC filename conventions.
 * Searches for files in directory <dir> of type <file_type>, possibly containing
   <job_name>, and returns the first match in the priority lists below.
   @param {string} dir Directory to search
   @param {string} job_name Root filename to search for
   @param {string} file_type File type to search for (can be "PRIMER", "D3PLOT", "T/HIS", "OTF")
   @returns {?string}
   @example
   let absolute_filename = find_lsdyna_files("C:/my/results/directory", "job_001", "D3PLOT");
 */
function find_lsdyna_files(dir, job_name, file_type) {
    let filters = [];
    let filename;

    switch (file_type) {
        case "PRIMER":
            filters = [
                new RegExp("^" + job_name + ".key$"),
                new RegExp("^" + job_name + ".k$"),
                new RegExp("^" + job_name + ".*.key$"),
                new RegExp("^" + job_name + ".*.k$"),
                new RegExp(".*.key$"),
                new RegExp(".*.k$"),
                new RegExp(".*.dyn$")
            ];

            break;

        case "D3PLOT":
            filters = [
                new RegExp("^" + job_name + ".ptf$"),
                new RegExp("^" + job_name + ".d3plot$"),
                new RegExp("^d3plot$"),
                new RegExp("^" + job_name + ".*.ptf$"),
                new RegExp("^" + job_name + ".*.d3plot$"),
                new RegExp(".*.ptf$"),
                new RegExp(".*.d3plot$"),
                new RegExp(".*.f(em)?z(ip)?$")
            ];
            break;

        case "T/HIS":
            filters = [
                new RegExp("^" + job_name + ".thf$"),
                new RegExp("^" + job_name + ".d3thdt$"),
                new RegExp("^d3thdt$"),
                new RegExp("^" + job_name + ".*.thf$"),
                new RegExp("^" + job_name + ".*.d3thdt$"),
                new RegExp(".*.thf$"),
                new RegExp(".*.d3thdt$"),
                new RegExp("^" + job_name + ".binout.*$"),
                new RegExp("^binout.*$"),
                new RegExp("^" + job_name + ".*.binout.*$"),
                new RegExp(".*.binout.*$")
            ];
            break;

        case "OTF":
            filters = [
                new RegExp("^" + job_name + ".otf$"),
                new RegExp("^" + job_name + ".d3hsp$"),
                new RegExp("^d3hsp$"),
                new RegExp("^" + job_name + ".*.otf$"),
                new RegExp("^" + job_name + ".*.d3hsp$"),
                new RegExp(".*.otf$"),
                new RegExp(".*.d3hsp$")
            ];
            break;

        default:
            LogError(`Unexpected <file_type = "${file_type}" in function find_lsdyna_files.`);
            Exit();
            break;
    }

    let filestore = [];
    for (let filter of filters) {
        let filelist = File.FindFiles(dir, `*`, false);
        for (let file of filelist) {
            if (filter.test(file) == true) {
                filestore.push(file);
            }
        }
        if (filestore.length > 0) {
            filestore.sort();
            /* Pick first matching file and strip off rest of path */
            filename = filestore[0].substring(
                Math.max(filestore[0].lastIndexOf("/"), filestore[0].lastIndexOf("\\")) + 1
            );
            break;
        }
    }

    if (filestore.length == 0) return null;

    let absolute_filename = dir + "/" + filename;

    return absolute_filename;
}

/**
 * Wrapper function to return the expanded variable value, handling the case where it doesn't exist
 * @param {Template} templ The REPORTER template object
 * @param {string} name Variable name you want to get value for
 * @returns {?string}
 */
function get_expanded_variable_value(templ, name) {
    let value = templ.GetVariableValue(name);
    if (value == null) {
        return null;
    } else {
        let expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));

        /* recursively expand variables until no more variables are found */
        while (expanded_value != value) {
            LogPrint(`Expanded variable ${name} from "${value}" -> "${expanded_value}"`);
            value = expanded_value;
            expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));
        }

        /** there is a bug where the templ.ExpandVariablesInString() does not handle slashes correctly
         * and accidentally puts double slashes in the path.
         * This is not generally a problem except when working with UNC file paths
         * The workaround is to check if it starts with 4 slashes then remove 2 of them
         */
        if (/^[\/]{4}/.test(expanded_value)) {
            LogPrint(`Removing 2 slashes from the expanded variable value: ${expanded_value}`);
            expanded_value = expanded_value.substring(2); // Remove first two slashes
        }

        return expanded_value;
    }
}

/**
 * Wrapper function for Template.GetVariableValue that also:
 * 1. Checks that "int" and "float" variables are valid
 * 2. Updates a status object with information about any missing or invalid variables
 * 3. Creates a variable for any missing variables with the value "Missing"
 * @param {object} status Status object, used to track any missing or invalid variables
 * @param {string} name Variable  - can pass with %{name}% or just {name}
 * @param {string} [type = "string"] Variable type. Must be "string", "int", or "float".
 * @param {boolean} [required = true] Whether or not the varialbe is required (true) or optional (false)
 * @returns {*} Variable value
 * @example
 * let head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC_SCORE`, "float");
 */
function get_variable_value(status, name, type = "string", required = true) {
    let match;
    /** if the name is wrapped in % symbols then strip them off as GetVariableValue relies on name without %s */
    if ((match = name.match(/%(.*)%/))) name = match[1];
    let templ = Template.GetCurrent();
    let value = templ.GetVariableValue(name);
    if (value == null) {
        if (required) {
            status.success = false;
            status.missing.push(name);

            /* Create variable with value "Missing" so that REPORTER doesn't complain about missing
             * variables. */
            let missing_var = new Variable(
                templ,
                name,
                "Requested variable is missing",
                `Missing`,
                "String",
                false,
                true
            );
        } else {
            /* If the variable was not required, create variable with value "N/A" so that REPORTER
             * doesn't complain about missing variables, and so that user can see it is not
             * required. */
            let optional_var = new Variable(
                templ,
                name,
                "Requested variable is missing but optional so mark as N/A",
                `N/A`,
                "String",
                false,
                true
            );
        }

        return value;
    }
    if (type == "string") {
        /* For string variables, just return value without any further checks */
        return value;
    } else if (type == "int" || type == "float") {
        /* For numeric values, check they are valid integers or floats */
        let parsed;
        if (type == "int") {
            parsed = Number.parseInt(value);
        }
        if (type == "float") {
            parsed = Number.parseFloat(value);
        }
        if (Number.isNaN(parsed)) {
            status.success = false;
            status.invalid.push(name);
        }
        return parsed;
    } else {
        /* We shouldn't get here so print error and exit */
        LogError(`Unexpected argument <type = ${type}> in function get_variable_value.`);
        Exit();
    }
}

/**
 * Prints warnings about missing and/or invalid REPORTER variables that have been requested during
 * score calculations.
 * @param {object} status Contains information about missing/invalid variables, expected in the form { success: false, missing: [...], invalid: [...] }
 * @param {string} descriptor String describing what is affected
 * @example
 * let body_region_label = `knee, femur and pelvis`;
 * let status = { success: true, missing: [], invalid: [] };
 * status.success = false;
 * status.missing.push("M1_DRIVER_FEMUR_COMPRESSION_EXCEEDENCE_SCORE");
 * status.invalid.push("M1_DRIVER_KNEE_COMPRESSION_SCORE");
 * if (!status.success) {
 *     warn_about_missing_or_invalid_variables(status, `M1 DRIVER ${body_region_label} score calculation`);
 * }
 */
function warn_about_missing_or_invalid_variables(status, descriptor) {
    if (status.missing && status.missing.length > 0) {
        LogWarning(`The following variables required for the ${descriptor} are missing:`);
        for (let name of status.missing) {
            LogPrint(name);
        }
    }
    if (status.invalid && status.invalid.length > 0) {
        LogWarning(`The following variables used in the ${descriptor} have invalid values:`);
        for (let name of status.invalid) {
            LogPrint(name);
        }
    }
}

/**
 * Returns a list of models based on the value of the %MODEL_LIST% variable.
 * Expect the value of %MODEL_LIST% to be in the form:
 *
 *     "M1, M2, M3"
 *
 *  Return null if missing
 *  @returns {?string[]}
 */
function get_model_list() {
    let templ = Template.GetCurrent();
    let models_list_str = templ.GetVariableValue("MODEL_LIST");
    if (models_list_str == null) {
        return null;
    } else {
        return models_list_str.split(/,\s*/);
    }
}

/**
 * Get the parent path from a directory or file path (i.e. strip off last / or \)
 * @param {String} path
 * @returns {String} parent
 */
function get_parent(path) {
    let path_index = Math.max(path.lastIndexOf("/"), path.lastIndexOf("\\"));
    let parent = path.substring(0, path_index);
    return parent;
}

/**
 * Function to check the validity of the batch arguments
 * @param {Object} argument_info holds check status of the arguments and any errors messages
 * @param {String|String[]} rep_var_names the REPORTER variable name
 * @param {Boolean} is_directory if the variable is a directory (it will try to create it if it does not exist)
 * @returns {Boolean} true if a valid variable was found
 * @example
 * let argument_info = { error_messages: [] };
 * let success = check_batch_arguments(argument_info, `OUTPUT_DIR`, true);
 */
function check_batch_arguments(argument_info, rep_var_names, is_directory = false) {
    if (typeof rep_var_names === "string") {
        rep_var_names = [rep_var_names];
    }
    if (rep_var_names instanceof Array === false) {
        throw Error(`rep_var_name must be a string or array of strings`);
    }
    for (let rep_var_name of rep_var_names) {
        try {
            let rep_var_val = get_expanded_variable_value(templ, rep_var_name);
            if (!rep_var_val || rep_var_val == "")
                throw Error(`${rep_var_name} REPORTER variable was not defined in the batch command.`);
            if (is_directory && !File.IsDirectory(rep_var_val)) {
                /** check if output directory exists and try and create it if not */
                LogPrint(`Output directory doesn't yet exist. Creating...`);
                let success = File.Mkdir(rep_var_val);
                if (success) {
                    LogPrint(`Created output directory: ${rep_var_val}`);
                } else {
                    throw Error(`Unable to create output directory: ${rep_var_val}`);
                }
            } else if (!File.Exists(rep_var_val)) throw Error(`${rep_var_name} "${rep_var_val}" does not exist.`);

            argument_info[rep_var_name] = true;
            /** no need to continue as we found a valid variable in the permutations*/
            return true;
        } catch (error) {
            argument_info.error_messages.push(error.message);
            argument_info[rep_var_name] = false;
        }
    }
    return false;
}
