/* Utils.Version was added for REPORTER in Version 21. So based on if this function is present or not we can check which version we are running */
if (typeof Utils.Version != "function") {
    let message = `This workflow is only available for Oasys REPORTER version 21 and above.`;
    if (Batch()) {
        LogError(message);
    } else {
        Window.Message("Workflow not available", message);
    }
    Exit();
}
/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
/** set the PASS/FAIL text on the results table on individual page */
var status = { success: true, missing: [], invalid: [] };
on_load_script();
/**
 * #SimVT REPORTER on load script
 *
 * Runs as soon as the template is opened.
 *
 * In batch mode, will proceed to generate if:
 *
 * 1. Settings file can be found from %SIMVT_SETTINGS%
 * 2. The settings file will then specify further requirements which must also be met e.g.
 *    a. The settings file will specify the type of expected test data
 *    b. The settings file will specify the number and type of expected sim data
 *    c. if the type is "CSV config" we require a config file e.g. %M1_CSV_CONFIG%
 *    d. for all other types (e.g. "CSV data", "ISO-MME" and "Workflow") we require just %<MODEL_TAG>_FILE% and we check it exists and the type matches what we expect
 *    e. for "Workflow" type we also need to check if the workflow data is available
 * 2. Valid %OUTPUT_DIR% defined or valid default (subdirectory can be created)
 * 3. Valid %RESULTS_DIR% defined or valid default containing results files
 *
 * Otherwise will terminate generation with a log error.
 *
 * In interactive mode, the script asks the user a series of questions:
 *
 * Q1. Do you have a SIMVT settings file that you would like to use? [YES]. If you don't have a settings file you can click NO and you will be able to create it later.
 * Q2. [Q1-YES] Select the SIMVT settings file that you would like to use -> %SIMVT_SETTINGS%
 * Q3. [Q2-Valid(%SIMVT_SETTINGS%)]: (alternatively go straigt to Q5.)
 *     1. Select the test file (%<TEST_MODEL_TAG>%) -> %TEST_FILE%
 *     1.a [%<TEST_MODEL_TAG>_TYPE% = "CSV config" ] Select the test file CSV config -> %TEST_FILE_CONFIG%
 *     2. Select the simulation file associated with <MODEL_TAG> -> %<MODEL_TAG>_FILE%
 *     2.a [%<MODEL_TAG>_TYPE% = "CSV config" ] Select the test file CSV config -> %TEST_FILE_CONFIG%
 *     Q3.2. (and conditionally Q3.2.a) are repeated N times where N is the number of simulations defined by %SIMVT_SETTINGS%
 * Q4. [Q1-NO]
 *     1. Select the test file -> %TEST_FILE%
 *     2. Select the simulation file >> %SIMULATION_FILES%
 *     3. Do you want to select any more simulation files?
 *        [YES] prompt user to select a file and append file to >> %SIMULATION_FILES% then repeat Q4.3.
 *        [NO] GOTO Q5.
 * Q5. SimVT will open in T/HIS so that you can create a settings file [OK]
 *
 * if we get here and have a valid settings file then we can begin processing results
 *
 *
 * If proceeding to template generation, the script will deactivate itelf to avoid entering an
 * infinite loop, and it also controls generation and (de)activation of various other items.
 */
function on_load_script() {
    let templ = Template.GetCurrent();

    /** Delete temporary variables */
    templ.DeleteTemporaryVariables();

    /** set the workflows directory variable if it is not defined - it should be defined if the template has been saved as an orr */
    set_workflows_dir_variable();

    /** generate the REPORTER item with helper functions so that functions in:
     *  %WORKFLOWS_DIR%\scripts\sim_vs_test_correlation\post\reporter\reporter_helper_functions.js
     * can be used in this script (like importing a module). Note we need to do this after set_workflows_dir_variable()
     * so that %WORKFLOWS_DIR% is defined*/
    generate_item(`helper_functions`);

    /** add %LSDYNA_RESULT_FILES% */

    let LSDYNA_RESULT_FILES = new Variable(templ, `LSDYNA_RESULT_FILES`, "", "");
    let JOB_CONTROL = new Variable(templ, `JOB_CONTROL`, "", "Run");

    /**
     * Function to check the validity of the batch arguments
     * @param {Object} argument_info holds check status of the arguments and any errors messages
     * @param {String} rep_var_name the REPORTER variable name
     * @param {Boolean} is_directory if the variable is a directory (it will try to create it if it does not exist)
     */
    function check_batch_arguments(argument_info, rep_var_name, is_directory = false) {
        try {
            let rep_var_val = get_expanded_variable_value(templ, rep_var_name);
            if (!rep_var_val || rep_var_val == "")
                throw Error(`${rep_var_name} REPORTER variable was not defined in the batch command.`);
            if (is_directory && !File.IsDirectory(rep_var_val)) {
                /** check if output directory exists and try and create it if not */
                LogPrint(`Output directory doesn't yet exist. Creating...`);
                let success = File.Mkdir(rep_var_val);
                if (success) {
                    LogPrint(`Created output directory: ${rep_var_val}`);
                } else {
                    throw Error(`Unable to create output directory: ${rep_var_val}`);
                }
            } else if (!File.Exists(rep_var_val)) throw Error(`${rep_var_name} "${rep_var_val}" does not exist.`);

            argument_info[rep_var_name] = true;
        } catch (error) {
            argument_info.error_messages.push(error.message);
            argument_info[rep_var_name] = false;
        }
    }

    try {
        if (Batch()) {
            /** defined error messages array */
            let argument_info = { error_messages: [] };
            /** first check all the possible batch arguments for validity */
            check_batch_arguments(argument_info, `OUTPUT_DIR`, true);
            check_batch_arguments(argument_info, `SIMVT_SETTINGS_FILE`);
            check_batch_arguments(argument_info, `SIMULATION_DATA_PATH`);
            check_batch_arguments(argument_info, `TEST_DATA_PATH`);

            /** then check for a valid combination */

            /** if output dir is invalid then we cannot proceed */
            if (!argument_info[`OUTPUT_DIR`]) throw Error(argument_info.error_messages.join(`\n`));
            /** if we have no settings file then both SIMULATION_DATA_PATH and TEST_DATA_PATH must be defined else we cannot proceed */
            if (
                !argument_info[`SIMVT_SETTINGS_FILE`] &&
                (!argument_info[`SIMULATION_DATA_PATH`] || !argument_info[`TEST_DATA_PATH`])
            )
                throw Error(argument_info.error_messages.join(`\n`));

            /** note that SIMULATION_DATA_PATH and TEST_DATA_PATH are set to "" by default in the template*/
        } else {
            let OUTPUT_DIR = new Variable(templ, `OUTPUT_DIR`, "", "");
        }
        let vcI_rep_var = new Variable(templ, `VALIDATION_CRITERION_1`, "Default", "-");

        /** Clear summary tables */
        clean_up_table();

        /* Create a fresh, empty AAW_job_control_variables.csv in case it has data from previous runs */
        initialise_job_control_variables_csv();

        /* To begin with, deactivate all #SIMVT items as we want them to only be generated when we explicityl call generate... */
        activate_all_simvt_items(false);

        /** The SimVT script runs T/HIS first and if that was successful then we finish script */
        generate_item(`#SIMVT T/HIS check and do assessment`);
        /** check the status of the JOB after generating the T/HIS item */
        check_job_status();

        /** try and parse the reporter JSON */
        parse_reporter_json();

        finish_script(true);
    } catch (error) {
        finish_script(false, error.message);
    }
}

/**
 * checks the values of JOB_CONTROL and JOB_MESSAGE and throws and error if their status represents an error
 */
function check_job_status() {
    /** update the JOB control status */
    generate_item(`#SIMVT REPORTER read job control`);

    /** parse the job control variables to determine if T/HIS was successful */
    let job_control = Variable.GetFromName(Template.GetCurrent(), "JOB_CONTROL");
    let job_message_var = Variable.GetFromName(Template.GetCurrent(), "JOB_MESSAGE");

    if (!job_control) throw Error(`%JOB_CONTROL% is not defined so status of T/HIS assessment is unknown.`);

    let msg = `%JOB_CONTROL% is ${job_control.value}`;
    if (job_message_var && job_message_var.value != "") msg = job_message_var.value;

    if (job_control.value != "Skip") throw Error(msg);
}

/* Delete AAW_job_control_variables.csv in case it has data from previous runs */
function initialise_job_control_variables_csv() {
    let templ = Template.GetCurrent();
    let reporter_temp = get_expanded_variable_value(templ, `REPORTER_TEMP`);
    if (reporter_temp == null || (reporter_temp != null && !File.IsDirectory(reporter_temp))) {
        finish_script(false, `Could not find REPORTER_TEMP directory.`);
    }
    let f_csv_name = `${reporter_temp}/AAW_job_control_variables.csv`;
    if (File.Exists(f_csv_name)) {
        let success = File.Delete(f_csv_name);
        if (!success) {
            finish_script(false, `Unable to delete existing variables file: ${f_csv_name}`);
        }
    }
    /* Replace it with an empty file so that `#SIMVT REPORTER read variables from PRIMER job control`
     * item always has something to read. */
    let f_csv = new File(f_csv_name, File.WRITE);
    f_csv.Close();
}

/**
 * @returns {Boolean}
 * @example
 * let success = get_output_dir();
 */
function get_output_dir() {
    let templ = Template.GetCurrent();

    let output_dir;
    if (Batch()) {
        /* First check to see if we can find a settings file.
         * We check SIMVT_SETTINGS_FILE first
         */
        output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);

        let valid_output_dir = false;
        if (output_dir != null && File.IsDirectory(output_dir)) {
            LogError(`${output_dir} is not a directory`);
            valid_output_dir = true;
        } else {
            if (output_dir == null) {
                LogPrint(`REPORTER Variable %OUTPUT_DIR% was not defined.`);
            } else {
                LogPrint(
                    `Settings file specified by REPORTER Variable %SIMVT_SETTINGS_FILE% could not be found: ${output_dir}`
                );
            }
        }

        if (!valid_output_dir) {
            return false;
        }
    } else {
        let ans = Window.Question(
            `Select output directory file`,
            `Select the directory where you want to save the report and outputs`,
            Window.OK | Window.CANCEL
        );
        if (ans == Window.CANCEL) {
            return false;
        }

        output_dir = Window.GetDirectory();

        if (!output_dir) {
            /* User clicked cancel */
            return false;
        }

        if (!File.IsDirectory(output_dir)) {
            LogError(`${output_dir} is not a directory`);
            return false;
        }

        set_output_file_variable(output_dir);

        return true;
    }
}

/**
 * Sets the REPORTER variables %OUTPUT_DIR%
 * The latter 2 are extracted from the settings file
 * specified keyword file.
 * @param {string} output_dir Absolute path and filename of settings file
 */
function set_output_file_variable(output_dir) {
    let templ = Template.GetCurrent();

    /* If output_dir is relative path, make it absolute path. We need to do this so that
     * any other variables that end up being derived from it (DEFAULT_DIR, RESULTS_DIR, OUTPUT_DIR,
     * etc.) are absolute paths too. When REPORTER generates an Oasys item, it sets -start_in to
     * the job file directory, so it gets confused if the relative paths we have defined are
     * relative to a different starting directory. Converting everything to absolute paths avoids
     * this problem. */
    output_dir = convert_to_absolute_path(output_dir, `%OUTPUT_DIR%`);

    let path_index = Math.max(output_dir.lastIndexOf("/"), output_dir.lastIndexOf("\\"));
    let filename = output_dir.substring(path_index + 1);
    let default_job = filename; //the name of the REPORT will be the same as the output folder by default

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * OUTPUT_DIR should be temporary; DEFAULT_DIR and DEFAULT_JOB are not temporary. */
    let output_dir_var = new Variable(templ, `OUTPUT_DIR`, `Output Directory`, output_dir, `Directory`, false, true);
    let default_dir_var = new Variable(
        templ,
        `DEFAULT_DIR`,
        `Reporter default directory`,
        output_dir,
        `Directory`,
        false,
        false
    );
    let default_job_var = new Variable(
        templ,
        `DEFAULT_JOB`,
        `Reporter default jobname`,
        default_job,
        `File(basename)`,
        false,
        false
    );
}

/**
 * Sets the REPORTER %WORKFLOW_DIR% variable - if it is already defined then use the existing value, provided it is valid.
 * Otherwise use the "%TEMPLATE_DIR%/../../" and if that is also not valid then prompt the user to select it until they selecct
 * a valid workflows dir.
 */
function set_workflows_dir_variable() {
    let templ = Template.GetCurrent();
    let workflows_dir;

    let workflows_dir_var = Variable.GetFromName(templ, "WORKFLOWS_DIR");
    if (workflows_dir_var && check_or_get_valid_workflows_dir(workflows_dir_var.value)) {
        return;
    }

    /**use the template dir if WORKFLOWS_DIR is not defined or valid */

    let template_dir = Variable.GetFromName(templ, "TEMPLATE_DIR").value;

    /** workflows directory is "%TEMPLATE_DIR%/../../" so get parent twice */
    workflows_dir = get_parent(get_parent(template_dir));

    /** get valid directory - keep showing window until user selects a valid workflows dir */
    workflows_dir = check_or_get_valid_workflows_dir(workflows_dir, true);

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * WORKFLOWS_DIR should not be temporary*/
    workflows_dir_var = new Variable(
        templ,
        `WORKFLOWS_DIR`,
        `Workflows Directory`,
        workflows_dir,
        `Directory`,
        false,
        false
    );
}

/**
 * this checks if the path is a valid workflow directory and if it is not it gets user to select a directory untill they select a valid one
 * @param {String} workflows_dir path
 * @param {Boolean} [selection_window = false] if true the user is prompted to select a workflow directory recurrsively until they get a valid one
 * @returns {?String} workflows_dir
 */
function check_or_get_valid_workflows_dir(workflows_dir, selection_window = false) {
    let msg;
    let expected_file_in_workflow_dir = `${workflows_dir}/scripts/automotive_assessments/modules/post/this/proxy.mjs`;

    if (!workflows_dir) {
        msg = `<workflows_dir> is undefined`;
    } else if (!File.IsDirectory(workflows_dir)) {
        msg = `"${workflows_dir}" does not exist`;
    } else if (!File.Exists(expected_file_in_workflow_dir)) {
        msg = `"${expected_file_in_workflow_dir}" does not exist so workflows_dir is invalid`;
    } else {
        return workflows_dir;
    }

    if (selection_window) {
        Window.Warning(
            `Select Workflows Directory`,
            `${msg}\nPlease select the workflows directory to use.`,
            Window.OK
        );
        workflows_dir = Window.GetDirectory();

        return check_or_get_valid_workflows_dir(workflows_dir, selection_window);
    }

    LogWarning(msg);
    return null;
}

/**
 * Common function for finishing the script.
 * Complete the generation of the template if we have all the required information.
 * @param {Boolean} can_generate Whether or not to generate the template
 * @param {string} [msg] Error message (provide when not generating)
 * @example
 * finish_script(Template.GetCurrent(), false, `Keyword file not provided`);
 */
function finish_script(can_generate, msg) {
    let templ = Template.GetCurrent();
    /* Complete the template generation if we have the required information,
     * otherwise end with message */
    if (can_generate) {
        /* Deactivate this script item to avoid entering an infinite loop */
        activate_master_page_item(`#SIMVT REPORTER on load script`, false);
        templ.Generate();
    } else {
        if (Batch()) LogError(msg);
        else {
            Window.Message(`Report automotation`, msg);
        }
    }

    /* Whether or not we generated, deactivate all the script items except this script item. */
    activate_all_simvt_items(false);
    activate_master_page_item(`#SIMVT REPORTER on load script`, true);

    /** change the template view to presentation mode and update before exiting the script*/
    templ.view = Reporter.VIEW_PRESENTATION;
    templ.Update();
    Exit();
}

/**
 * Attempt to parse the reporter json. Throw an Error if we encounter an issue (e.g. the reporter_json_file_path does not exist)
 */

function parse_reporter_json() {
    let reporter_temp = Variable.GetFromName(Template.GetCurrent(), "REPORTER_TEMP").value;

    let reporter_json_file_path = `${reporter_temp}/reporter.json`;
    if (File.Exists(reporter_json_file_path)) {
        let reporter_json_file = new File(reporter_json_file_path, File.READ);

        let line;
        let lines = [];

        //@ts-ignore
        while ((line = reporter_json_file.ReadLongLine()) != File.EOF) {
            lines.push(line);
        }

        reporter_json_file.Close();

        /** @type {REPORTER_JSON} */
        let reporter_json = JSON.parse(lines.join(`\n`));

        /** set the output directory based on the reporter json property*/
        set_output_file_variable(reporter_json.output_directory);

        /** create the variables for populating the results tables*/
        create_reporter_variables(reporter_json);

        /** create the pages if they are missing */
        create_and_update_pages(reporter_json);
    } else {
        throw Error(`${reporter_json_file_path} does not exist so cannot generate template.`);
    }
}

function create_and_update_pages(reporter_json) {
    /** we need a page for each 'selected' channel (or in the case of the EuroNCAP vtc for each protocol channel) */
    let templ = Template.GetCurrent();

    /** one page is expected for each row of the MAIN_SUMMARY_TABLE item
     * alternatively, one page is expected for each sibling group of reporter_json.correlations or protocol_channel_tags
     */

    let expected_pages = {};

    if (reporter_json.protocol_channel_tags) {
        /** we are in protocol mode */
        for (let iso_channel in reporter_json.protocol_channel_tags) {
            /**
             * split protocol_channel_tags into simvt_page objects
             */
            if (iso_channel.length != 16) {
                LogWarning(`${iso_channel} is not a valid iso channnel so will be skipped`);
                continue;
            }

            let direction = iso_channel.substring(14, 15);
            let name = iso_channel.substring(0, 14) + "_" + iso_channel.substring(15, 16);
            let exists = reporter_json.protocol_channel_tags[iso_channel];

            if (!expected_pages[name]) expected_pages[name] = [];
            expected_pages[name].push({ channel: iso_channel, direction: direction, exists: exists });
        }
    }

    let page_xyz = get_page_by_name("PLACEHOLDER_XYZ", false);
    let page_1d = get_page_by_name("PLACEHOLDER_1D", false);

    let num_channels_processed = 0;

    let results_pages = [];

    /** convert object to an array */
    for (let page_name in expected_pages) {
        let page = get_page_by_name(page_name, false);
        results_pages.push({ name: page_name, num_channels: expected_pages[page_name].length, page: page });
        if (page instanceof Page) {
            switch (expected_pages[page_name].length) {
                case 1:
                    page_1d = page;
                    break;
                case 3:
                    page_xyz = page;
                    break;
            }
        }
    }

    /** create the page if it is missing */
    for (let i = 0; i < results_pages.length; i++) {
        if (!(results_pages[i].page instanceof Page)) {
            switch (results_pages[i].num_channels) {
                case 1:
                    results_pages[i].page = page_1d.Duplicate(i + 3);
                    break;
                case 2: //use page_xyz for 2 channels for now. TODO update this to use a page specific to 2 channels
                case 3:
                    results_pages[i].page = page_xyz.Duplicate(i + 3);
                    break;
                default:
                    throw Error(`PLACEHOLDER page with ${results_pages[i].num_channels} channels not supported`);
            }
            /** set the new page name */
            results_pages[i].page.name = results_pages[i].name;
        }
    }

    /** update all pages */
    for (let page_data of results_pages) {
        let page = page_data.page;
        let channels = expected_pages[page_data.page.name];

        /** set the heading */
        let heading_item = get_item_by_name("HEADING", page);
        heading_item.text = page_data.page.name;

        /** set the table cells*/
        let summary_table_item = get_item_by_name("SUMMARY_TABLE", page);

        let column_order = [
            "SIM_MODEL_TAG",
            "SIM_CHANNEL",
            "REF_MODEL_TAG",
            "REF_CHANNEL",
            "CORRIDOR_RATING",
            "SLOPE_RATING",
            "PHASE_RATING",
            "MAGNITUDE_RATING",
            "TOTAL_SIGNAL_RATING",
            "MAX_AMPLITUDE",
            "WEIGHT"
        ];

        if (page_data.num_channels == 1) {
            column_order.push("TOTAL_SIGNAL_RATING");
        } else {
            column_order.push("WEIGHTED_SIGNAL_RATING");
        }

        for (let i = 0; i < channels.length; i++) {
            let col = 0;
            /** start at the second row */
            let row = i + 2;
            let channel = channels[i].channel;
            let axis = channels.length == 1 ? "1D" : channels[i].direction;
            for (let column of column_order) {
                /** the reporter variable is the channel code and the results CSV column header with an underscore between */
                let cell_text = `%${channel}_${column}%`;
                /** get the value so that if it does not exist it will be created with "Missing" value */

                let value = get_variable_value(status, cell_text);
                if (!value || value == "Missing") {
                    switch (column) {
                        case "SIM_MODEL_TAG":
                        case "REF_MODEL_TAG":
                            break;
                        case "SIM_CHANNEL":
                        case "REF_CHANNEL":
                            let v = new Variable(templ, `${channel}_${column}`, `Channel code`, channel);
                            break;
                    }
                }
                /** set the cell on the summary table of the channel page */
                set_cell_text(summary_table_item, row, col, cell_text);
                /** add the  text to the detailed results table too*/
                update_detailed_results_pages(num_channels_processed, col, cell_text);
                col++;
            }
            num_channels_processed++;

            /** now change the placeholders for images to actual image items
             * there is no "Convert Into" method so we create a new Item.IMAGE and set the dimensions
             * to match the placeholder item, then delete the placeholder item
             */
            let placeholder_item, image_item, placeholder_name;
            try {
                placeholder_name = `${axis}_PLACEHOLDER`;
                placeholder_item = get_item_by_name(placeholder_name, page);
            } catch (error) {}
            try {
                image_item = get_item_by_name(channel.substring(2), page);
            } catch (error) {}

            /** check if the item exists */

            try {
                let variable_sim = Variable.GetFromName(templ, `${channel}_${column_order[0]}`);
                try {
                    if (!variable_sim || variable_sim.value == `Missing`) {
                        if (placeholder_item) placeholder_item.text = `Missing ${channel} data`;
                        continue;
                    }
                } catch (error) {}

                /** convert the description to the image name */
                let filename = variable_sim.description.replace(/[\(\)\|\*]/g, "").replace(/\s+/g, "_");
                let img_file_txt = `%OUTPUT_DIR%/${filename}_Corridor.png`;
                let img_file_path = templ.ExpandVariablesInString(img_file_txt);

                if (!File.Exists(img_file_path)) {
                    LogWarning(`Image file ${img_file_path} does not exist so it will not be generated`);
                    continue;
                }

                if (placeholder_item) {
                    if (!image_item) {
                        image_item = new Item(
                            page,
                            Item.IMAGE_FILE,
                            channel,
                            placeholder_item.x,
                            placeholder_item.x2,
                            placeholder_item.y,
                            placeholder_item.y2
                        );
                    }
                } else if (!image_item) {
                    LogWarning(
                        `Expected either ${axis}_PLACEHOLDER Item or ${channel.substring(2)} on page ${
                            page.name
                        }, but neither exists`
                    );
                    continue;
                }

                /** ensure that the border line colour is off for the images */
                image_item.lineColour = Colour.None();

                /** check if the item exists */

                if (File.Exists(img_file_path)) {
                    image_item.file = img_file_txt;
                    if (placeholder_item) delete_item_by_name(placeholder_name, page);
                    image_item.Generate();
                }
            } catch (error) {
                LogError(error.message);
            }
        }
        /** update summary page */
        set_summary_table(channels, page);
    }

    /** update VC1 table */
    update_vc1_table();

    /** delete placeholder pages */
    delete_pages_by_name("PLACEHOLDER_XYZ");
    delete_pages_by_name("PLACEHOLDER_1D");

    /**
     * @typedef {Object} simvt_page
     * @property {String} name
     * @property {num_channels} channels
     */
}

function update_detailed_results_pages(row, col, cell_text) {
    let page_detailed_results = get_page_by_name("DETAILED_RESULTS");
    let summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", page_detailed_results);

    let num_header_rows = 2;
    let remainder = row % (summary_table_item.rows - num_header_rows);
    let table_index = (row - remainder) / (summary_table_item.rows - num_header_rows);

    // no suffix if table_index == 0 otherwise it is added with underscore e.g. DETAILED_RESULTS_1;
    let page_suffix = table_index ? `_${table_index}` : "";

    let new_page_detailed_results = get_page_by_name(`DETAILED_RESULTS${page_suffix}`, false);
    if (!new_page_detailed_results) {
        new_page_detailed_results = page_detailed_results.Duplicate();
        /** det the new page name so that it can be found the next time */
        new_page_detailed_results.name = `DETAILED_RESULTS${page_suffix}`;
        let header = get_item_by_name("Detailed Results", new_page_detailed_results);
        header.text = "Detailed Results (Continued)";
        summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", new_page_detailed_results);
        //todo clear the table
        for (let r = num_header_rows; r < summary_table_item.rows; r++) {
            for (let c = 0; c < summary_table_item.columns; c++) {
                set_cell_text(summary_table_item, row, col, "");
            }
        }
    }
    summary_table_item = get_item_by_name("DETAILED_RESULTS_TABLE", new_page_detailed_results);

    /** set the summary page table*/
    set_cell_text(summary_table_item, remainder + num_header_rows, col, cell_text);
}

function update_vc1_table() {
    for (let page_name of [`Summary`]) {
        let vcI_table_item = get_named_item_on_named_page(`VALIDATION_CRITERION_1_TABLE`, page_name);

        set_cell_text(vcI_table_item, 0, 1, `%VALIDATION_CRITERION_1%`);
        set_cell_conditional_formatting(vcI_table_item, 0, 1, false, true);
    }
}

/**
 * set the results table value to a variable that will be one of "Missing"|"PASS"|"FAIL"|"PASS"|"(FAIL)"|"(PASS)"
 * @param {Page} results_page
 * @param {String|Number} sensor_score
 * @param {Boolean} mandatory if not mandatory reuslt will be "(FAIL)"|"(PASS)"
 * @returns {String} result - the value the results table cell will display
 */
function set_results_table(results_page, sensor_score, mandatory) {
    let templ = Template.GetCurrent();
    let vcI_var_name = `${results_page.name}_VALIDATION_CRITERION_I`;

    let result = `Missing`;

    /** if sensor_score is a number then evaluate pass/fail result else result will be "Missing"*/
    if (!isNaN(+sensor_score)) {
        if (mandatory) result = +sensor_score < 0.5 ? "FAIL" : "PASS";
        else result = +sensor_score < 0.5 ? "(FAIL)" : "(PASS)";
    }
    /** create a REPORTER variable - this will overwrite it if it already exists */
    let vcI = new Variable(
        templ,
        vcI_var_name,
        `Validation criterion I PASS/FAIL/Missing for ${results_page.name}`,
        result,
        "String"
    );

    let results_table_item = get_item_by_name(`RESULTS_TABLE`, results_page);

    set_cell_text(results_table_item, 0, 1, `%${vcI_var_name}%`);
    set_cell_conditional_formatting(results_table_item, 0, 1, false, mandatory);

    if (mandatory) {
        /** this is initialised to "PASS" at the start but it will only stay as pass if result is always "PASS" */
        let vcI_rep_var = Variable.GetFromName(templ, `VALIDATION_CRITERION_1`);
        if (vcI_rep_var) {
            /** if the current value of vcI_rep_var is "FAIL" then return as this is the worst outcome...
             * a fail of one mandatory sensor is still a fail even if data is missing for other sensors */
            if (vcI_rep_var.value == "FAIL") return;
            if (result != "PASS" || vcI_rep_var.description == "Default") {
                /** only set the value if the current sensor does not pass (i.e. "FAIL" or "Missing") */
                vcI_rep_var.value = result;
                vcI_rep_var.description == "summary of madatory sensor scores";
                return;
            }
        }
    }
}

/**
 *
 * @param {Item} table
 * @param {Number} row
 * @param {Number} col
 * @param {Boolean} numeric if expected values are numbers (e.g. 0.5) or strings (e.g. "PASS")
 * @param {Boolean} mandatory if the condition is mandatory
 */
function set_cell_conditional_formatting(table, row, col, numeric, mandatory) {
    /* Euro NCAP colours */
    const GREEN = Colour.RGB(38, 155, 41); //"#269b29";
    const GREEN_MUTED = Colour.RGB(202, 255, 202); //"#caffca";
    const YELLOW = "#ffcc00";
    const ORANGE = Colour.RGB(255, 151, 0); //"#ff9700";
    const BROWN = "#753f2a";
    const RED = Colour.RGB(224, 7, 0); //"#e00700";
    const RED_MUTED = Colour.RGB(224, 138, 136); //"#e08a88";

    let default_props = table.GetCellProperties(row, col);

    let mandatory_text = mandatory ? "_MANDATORY" : "";
    let conditions = [
        {
            name: `MISSING`,
            type: Reporter.CONDITION_EQUAL_TO,
            value: "Missing",
            fillColour: ORANGE,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        }
    ];
    if (numeric) {
        conditions.push({
            name: `VCI_PASS${mandatory_text}`,
            type: Reporter.CONDITION_GREATER_THAN,
            value: "0.5",
            fillColour: mandatory ? GREEN : GREEN_MUTED,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
        conditions.push({
            name: `VCI_FAIL${mandatory_text}`,
            type: Reporter.CONDITION_LESS_THAN,
            value: "0.5",
            fillColour: mandatory ? RED : RED_MUTED,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
        conditions.push({
            name: `VCI_FAIL${mandatory_text}`,
            type: Reporter.CONDITION_EQUAL_TO,
            value: "0.5",
            fillColour: mandatory ? RED : RED_MUTED,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
    } else {
        conditions.push({
            name: `PASS`,
            type: Reporter.CONDITION_EQUAL_TO,
            value: "PASS",
            fillColour: GREEN,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
        conditions.push({
            name: `FAIL`,
            type: Reporter.CONDITION_EQUAL_TO,
            value: "FAIL",
            fillColour: RED,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
        conditions.push({
            name: `OPTIONAL PASS`,
            type: Reporter.CONDITION_EQUAL_TO,
            value: "(PASS)",
            fillColour: GREEN_MUTED,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
        conditions.push({
            name: `OPTIONAL FAIL`,
            type: Reporter.CONDITION_EQUAL_TO,
            value: "(FAIL)",
            fillColour: RED_MUTED,
            fontName: default_props.fontName,
            fontSize: default_props.fontSize,
            fontStyle: default_props.fontStyle,
            justify: default_props.justify,
            textColour: default_props.textColour
        });
    }

    let num_conditions = Math.max(default_props.conditions, conditions.length);
    for (let i = 0; i < num_conditions; i++) {
        if (i < conditions.length) table.SetCondition(i, row, col, conditions[i]);
        else {
            /** clear conditions that were already set that we no longer need */
            if (Utils.Version() < 22) {
                //@ts-ignore
                table.SetCondition(i, row, col, {});
            } else {
                table.RemoveCondition(i, row, col);
            }
        }
    }
}

/**
 *
 * @param {Object[]} channels
 * @param {Page} results_page
 * @returns {?String} sensor_heading
 */
function set_summary_table(channels, results_page) {
    let page_name = "Summary";

    // let x_channel, y_channel, z_channel;

    // for (let channel of channels){
    //     switch (channel.direction.toUpperCase()) {
    //         case "X":

    //             break;

    //         default:
    //             break;
    //     }
    // }

    let x_channel = channels[0] ? channels[0].channel : null;
    let y_channel = channels[1] ? channels[1].channel : null;
    let z_channel = channels[2] ? channels[2].channel : null;
    let summary_page = get_page_by_name(page_name);

    /** set the table cells*/
    let summary_table_item = get_item_by_name("MAIN_SUMMARY_TABLE", summary_page);

    let default_na = "";
    let column_order = [
        x_channel ? `%${x_channel}_TOTAL_SIGNAL_RATING%` : default_na,
        x_channel ? `%${x_channel}_MAX_AMPLITUDE%` : default_na,
        y_channel ? `%${y_channel}_TOTAL_SIGNAL_RATING%` : default_na,
        y_channel ? `%${y_channel}_MAX_AMPLITUDE%` : default_na,
        z_channel ? `%${z_channel}_TOTAL_SIGNAL_RATING%` : default_na,
        z_channel ? `%${z_channel}_MAX_AMPLITUDE%` : default_na
    ];

    /** if we only have 1 channel then the WEIGHTED_SIGNAL_RATING is blank so we need to (re)use total signal rating */
    let sensor_score_rep_var_name;
    if (channels.length == 1) sensor_score_rep_var_name = column_order[0];
    else {
        let x_iso_score = get_variable_value(status, `${x_channel}_TOTAL_SIGNAL_RATING`);
        let y_iso_score = get_variable_value(status, `${y_channel}_TOTAL_SIGNAL_RATING`);
        let z_iso_score = get_variable_value(status, `${z_channel}_TOTAL_SIGNAL_RATING`);

        if (x_iso_score == "Missing" || y_iso_score == "Missing" || z_iso_score == "Missing") {
            /** if any of the scores are missing then we need to set all the weighted signal rating to missing */
            let x_signal_weighting = new Variable(
                Template.GetCurrent(),
                `${x_channel}_WEIGHTED_SIGNAL_RATING`,
                `Sensor score`,
                "Missing"
            );
            let y_signal_weighting = new Variable(
                Template.GetCurrent(),
                `${y_channel}_WEIGHTED_SIGNAL_RATING`,
                `Sensor score`,
                "Missing"
            );
            let z_signal_weighting = new Variable(
                Template.GetCurrent(),
                `${z_channel}_WEIGHTED_SIGNAL_RATING`,
                `Sensor score`,
                "Missing"
            );
        }

        sensor_score_rep_var_name = x_channel ? `%${x_channel}_WEIGHTED_SIGNAL_RATING%` : default_na;
    }

    column_order.push(sensor_score_rep_var_name);

    for (let row = 2; row < summary_table_item.rows; row++) {
        let col = 2; /** skip columns 1 and 2 */
        let row_iso_code = get_cell_text(summary_table_item, row, 1);
        let re_string = row_iso_code.replace(/_/g, ".");
        let re = new RegExp(re_string);
        LogPrint(`checking if ${x_channel} matches ${re_string} `);
        if (!re.test(x_channel)) continue;

        for (let column of column_order) {
            LogPrint(`setting value of column ${column}, row ${row + 1} ${col + 1} `);
            // let cell_value = "N/A";
            // if (get_expanded_variable_value(Template.GetCurrent(), column)) column = `%${column}%`;
            set_cell_text(summary_table_item, row, col, column);
            col++;
        }

        let sensor_score = get_variable_value(status, sensor_score_rep_var_name, "float");
        let mandatory =
            get_cell_text(summary_table_item, row, summary_table_item.columns - 1).toLocaleUpperCase() == "YES";
        set_results_table(results_page, sensor_score, mandatory);

        /** the penultimate column is the sensor score column */
        set_cell_conditional_formatting(summary_table_item, row, summary_table_item.columns - 2, true, mandatory);

        let sensor_heading = get_cell_text(summary_table_item, row, 0);
        return sensor_heading;
    }

    return null;
}

/**
 * create reporter variables from the results CSV defined on the reporter_json
 * TODO - change this to use the reporter_json rather than results.csv
 * @param {REPORTER_JSON} reporter_json
 * @returns {Object} channel_strings_obj
 */
function create_reporter_variables(reporter_json) {
    const LOCAL_DEBUG = false;
    let templ = Template.GetCurrent();
    let temp_var;

    /** try extracting head excursion results data */
    try {
        if (reporter_json.he_data) {
            let t_end = reporter_json.he_data[reporter_json.sim_model_tag].t_end;
            let t_max = reporter_json.he_data[reporter_json.sim_model_tag].t_max;
            let max_he = reporter_json.he_data[reporter_json.sim_model_tag].max_he;
            let pass = reporter_json.he_data[reporter_json.sim_model_tag].pass;

            temp_var = new Variable(templ, `SIM_T_END`, `simulation end time (s)`, t_end, "Number", false, true);

            temp_var = new Variable(
                templ,
                `SIM_T_MAX_HE`,
                `simulation time of maximum head excursion (s)`,
                t_max,
                "Number",
                false,
                true
            );

            temp_var = new Variable(
                templ,
                `SIM_MAX_HE`,
                `maximum head excursion distance (m)`,
                max_he,
                "Number",
                false,
                true
            );

            temp_var = new Variable(
                templ,
                `SIM_STATUS`,
                `PASS/FAIL status of simulation pass if t_end > (tmax * 1.2)`,
                pass ? "PASS" : "FAIL",
                "String",
                false,
                true
            );
        }
    } catch (error) {
        LogError(error.message);
    }

    /** get the vairables so that if they are missing they will be created and set to "Missing" */
    get_variable_value(status, `SIM_T_MAX_HE`);
    get_variable_value(status, `SIM_T_END`);
    get_variable_value(status, `SIM_MAX_HE`);
    get_variable_value(status, `SIM_STATUS`);

    /** set REPORTER variables for sim and test file paths */

    Variable.GetFromName(templ, `SIMULATION_DATA_PATH`).value = reporter_json.sim_file_path;
    Variable.GetFromName(templ, `TEST_DATA_PATH`).value = reporter_json.ref_file_path;

    // for (let correlation of reporter_json.correlations){

    //     let temp_var = new Variable(
    //         templ,
    //         `${iso_14}_${prop}`,
    //         run_id,
    //         tag_channel_obj[prop],
    //         "String",
    //         false,
    //         true
    //     );

    //     tag_channel_obj["SIM_MODEL_TAG"] = match[1];
    //     tag_channel_obj["REF_MODEL_TAG"] = match[2];
    //     tag_channel_obj["SIM_CHANNEL"] = match[3];
    //     tag_channel_obj["REF_CHANNEL"] = match[3];

    // }

    let results_csv_path = reporter_json.results_csv;

    if (!File.Exists(results_csv_path)) {
        throw Error(`Cannot read results as they do not exist here: "${results_csv_path}"`);
    }
    let results_csv = new File(results_csv_path, File.READ);

    let headers = results_csv.ReadLongLine().split(`,`);

    let line;
    let channel_strings_obj = {};
    let line_count = 1;
    //@ts-ignore
    while ((line = results_csv.ReadLongLine()) != File.EOF) {
        line_count++;
        /** split up the line by commas */
        let values = line.split(`,`);
        let temp_obj = {};
        let run_id, method;
        for (let col = 0; col < headers.length; col++) {
            let header = headers[col];
            let value = values[col];

            /** we skip adding headers with text to the object */
            switch (header) {
                case "RUN_ID":
                    run_id = value;
                    continue;
                case "METHOD":
                    method = value;
                    continue;
                case "ISO_RATING_MEANING":
                    continue;
            }

            temp_obj[header] = value;
        }

        try {
            if (!run_id) throw Error(`No RUN_ID on line ${line_count} so it will be skipped `);

            if (!method) throw Error(`No METHOD on line ${line_count} so it will be skipped `);

            /** now we create the temporary variables */

            let re_same_channel = /\(([^\(\)]*) vs (.*)\) (.*)/;
            let re_diff_channel = /\((.*)\) (.*) vs \((.*)\) (.*)/;

            let match;
            let tag_channel_obj = {};
            if ((match = run_id.match(re_same_channel))) {
                tag_channel_obj["SIM_MODEL_TAG"] = match[1];
                tag_channel_obj["REF_MODEL_TAG"] = match[2];
                tag_channel_obj["SIM_CHANNEL"] = match[3];
                tag_channel_obj["REF_CHANNEL"] = match[3];
            } else if ((match = run_id.match(re_diff_channel))) {
                tag_channel_obj["SIM_MODEL_TAG"] = match[1];
                tag_channel_obj["REF_MODEL_TAG"] = match[3];
                tag_channel_obj["SIM_CHANNEL"] = match[2];
                tag_channel_obj["REF_CHANNEL"] = match[4];
            } else {
                throw Error(`RUN_ID "${run_id}" did not match the expected pattern.`);
            }

            let sim_channel = tag_channel_obj["SIM_CHANNEL"];
            let iso_14 = sim_channel; //sim_channel.substring(sim_channel.length - 14);

            if (!channel_strings_obj.hasOwnProperty(iso_14.substring(0, iso_14.length - 2))) {
                channel_strings_obj[iso_14.substring(0, iso_14.length - 2)] = [];
            }
            channel_strings_obj[iso_14.substring(0, iso_14.length - 2)].push(iso_14.substring(iso_14.length - 2));

            for (let prop in tag_channel_obj) {
                if (LOCAL_DEBUG)
                    LogWarning(`tag_channel_obj[${prop}] = ${tag_channel_obj[prop]}, Creating %${iso_14}_${prop}%`);
                /** we need to convert the csv contents to REPORTER variables */
                let temp_var = new Variable(
                    templ,
                    `${iso_14}_${prop}`,
                    run_id,
                    tag_channel_obj[prop],
                    "String",
                    false,
                    true
                );
            }

            for (let header in temp_obj) {
                if (LOCAL_DEBUG)
                    LogWarning(`temp_obj[${header}] = ${temp_obj[header]}, Creating %${iso_14}_${header}%`);
                /** we need to convert the csv contents to REPORTER variables */
                let temp_var = new Variable(
                    templ,
                    `${iso_14}_${header}`,
                    run_id,
                    temp_obj[header],
                    "Number",
                    false,
                    true
                );
                temp_var.precision = 3;
                temp_var.format = Variable.FORMAT_FLOAT;
            }
        } catch (error) {
            LogError(error.toString());
        }
    }

    results_csv.Close();

    return channel_strings_obj;
}

/**
 * Activates or deactivates #activate_normal_page_simvt_items items on all pages (master page
 * and normal pages) so they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_all_simvt_items(false);
 */
function activate_all_simvt_items(active) {
    activate_master_page_simvt_items(active);
    activate_normal_page_simvt_items(active);
}

/**
 * Activates or deactivates #SIMVT items on normal pages so that
 * they are only generated when required.
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_normal_page_simvt_items(false);
 */
function activate_normal_page_simvt_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let hash_item_str = `#SIMVT`;

    /* Search for relevant items on other pages */
    let pages = templ.GetAllPages();
    for (let page of pages) {
        let items = page.GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, hash_item_str.length) == hash_item_str) {
                /* (De)activate all #SIMVT items */
                item.active = active;
            }
        }
    }
}

/**
 * Activates or deactivates #SIMVT master page items so that they
 * are only generated once (they sit on the Master page so would be generated for every page otherwise).
 * @param {boolean} active Whether to activate or deactivate the items
 * @example
 * activate_master_page_simvt_items(false);
 */
function activate_master_page_simvt_items(active) {
    let templ = Template.GetCurrent();

    /* We will search for items with names beginning with an identifiable string */
    let hash_item_str = `#SIMVT`;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let items = master.GetAllItems();

    for (let item of items) {
        if (item.name.substring(0, hash_item_str.length) == hash_item_str) {
            /* (De)activate all #SIMVT items */
            item.active = active;
        }
    }
}

/**
 * Activates or deactivates a specific item on the Master page so that it is only generated when
 * required.
 * @param {string} item_name The item name (the first item matching this name will be actioned)
 * @param {boolean} active Whether to activate or deactivate the item
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * activate_master_page_item(`#SIMVT REPORTER on load script`, false);
 */
function activate_master_page_item(item_name, active, expected = true) {
    let templ = Template.GetCurrent();

    /* Syntax for log messages */
    let de = `de`;
    if (active) de = ``;

    /* Search for relevant items on the Master page */
    let master = templ.GetMaster();

    let match = false;
    let items = master.GetAllItems();
    for (let item of items) {
        /* Deactivate this script item */
        if (item.name.substring(0, item_name.length) == item_name) {
            item.active = active;
            match = true;
            LogPrint(`Successfully ${de}activated master page item "${item_name}".`);
            break;
        }
    }

    if (!match && expected) {
        LogError(`Could not find master page item "${item_name}" to ${de}activate it.`);
        Exit();
    }
}

/**
 * Finds the first item in the template with the specified <item_name> and generates it.
 * Looks at the master page first, then all the normal pages.
 * @param {string} item_name The name of the item to be generated
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @example
 * generate_item(`#SIMVT REPORTER read variables from PRIMER job control`);
 */
function generate_item(item_name, expected = true) {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    pages.unshift(template.GetMaster());
    let match = false;
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name == item_name) {
                LogPrint(`Found item "${item_name}" on page ${p + 1}. Generating...`);
                match = true;
                /* Activate the item before generating it, then return it to its original status */
                let active = item.active;
                item.active = true;
                item.Generate();
                item.active = active;
                break;
            }
        }
        if (match) break;
    }
    if (!match && expected) {
        LogError(`Could not find item "${item_name}" in order to generate it.`);
        Exit();
    }
}

/**
 * Finds the first item in the template with the specified <item_name> and returns it.
 * Looks at the master page first, then all the normal pages.
 * @param {string} item_name The name of the item to be generated
 * @param {boolean} [expected = true] If true (default), will exit with an error if the requested item cannot be found
 * @returns {?Item}
 * @example
 * let item = get_item(`#SIMVT REPORTER read variables from PRIMER job control`);
 */
function get_item(item_name, expected = true) {
    let template = Template.GetCurrent();
    /* Loop through all pages in the template, searching for relevant items */
    let pages = template.GetAllPages();
    /* Add Master page to front so we consider it first */
    pages.unshift(template.GetMaster());
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name == item_name) {
                LogPrint(`Found item "${item_name}" on page ${p + 1}: "${pages[p].name}". `);
                return item;
            }
        }
    }
    if (expected) {
        LogError(`Could not find item "${item_name}" in order to generate it.`);
        Exit();
    }
    return null;
}

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}

/**
 * Searches a directory for filenames matching either Arup or LSTC filename conventions.
 * Searches for files in directory <dir> of type <file_type>, possibly containing
   <job_name>, and returns the first match in the priority lists below.
   @param {string} dir Directory to search
   @param {string} job_name Root filename to search for
   @param {string} file_type File type to search for (can be "PRIMER", "D3PLOT", "T/HIS", "OTF")
   @returns {?string}
   @example
   let absolute_filename = find_lsdyna_files("C:/my/results/directory", "job_001", "D3PLOT");
 */
function find_lsdyna_files(dir, job_name, file_type) {
    let filters = [];
    let filename;

    switch (file_type) {
        case "PRIMER":
            filters = [
                new RegExp("^" + job_name + ".key$"),
                new RegExp("^" + job_name + ".k$"),
                new RegExp("^" + job_name + ".*.key$"),
                new RegExp("^" + job_name + ".*.k$"),
                new RegExp(".*.key$"),
                new RegExp(".*.k$"),
                new RegExp(".*.dyn$")
            ];

            break;

        case "D3PLOT":
            filters = [
                new RegExp("^" + job_name + ".ptf$"),
                new RegExp("^" + job_name + ".d3plot$"),
                new RegExp("^d3plot$"),
                new RegExp("^" + job_name + ".*.ptf$"),
                new RegExp("^" + job_name + ".*.d3plot$"),
                new RegExp(".*.ptf$"),
                new RegExp(".*.d3plot$"),
                new RegExp(".*.f(em)?z(ip)?$")
            ];
            break;

        case "T/HIS":
            filters = [
                new RegExp("^" + job_name + ".thf$"),
                new RegExp("^" + job_name + ".d3thdt$"),
                new RegExp("^d3thdt$"),
                new RegExp("^" + job_name + ".*.thf$"),
                new RegExp("^" + job_name + ".*.d3thdt$"),
                new RegExp(".*.thf$"),
                new RegExp(".*.d3thdt$"),
                new RegExp("^" + job_name + ".binout.*$"),
                new RegExp("^binout.*$"),
                new RegExp("^" + job_name + ".*.binout.*$"),
                new RegExp(".*.binout.*$")
            ];
            break;

        case "OTF":
            filters = [
                new RegExp("^" + job_name + ".otf$"),
                new RegExp("^" + job_name + ".d3hsp$"),
                new RegExp("^d3hsp$"),
                new RegExp("^" + job_name + ".*.otf$"),
                new RegExp("^" + job_name + ".*.d3hsp$"),
                new RegExp(".*.otf$"),
                new RegExp(".*.d3hsp$")
            ];
            break;

        default:
            LogError(`Unexpected <file_type = "${file_type}" in function find_lsdyna_files.`);
            Exit();
            break;
    }

    let filestore = [];
    for (let filter of filters) {
        let filelist = File.FindFiles(dir, `*`, false);
        for (let file of filelist) {
            if (filter.test(file) == true) {
                filestore.push(file);
            }
        }
        if (filestore.length > 0) {
            filestore.sort();
            /* Pick first matching file and strip off rest of path */
            filename = filestore[0].substring(
                Math.max(filestore[0].lastIndexOf("/"), filestore[0].lastIndexOf("\\")) + 1
            );
            break;
        }
    }

    if (filestore.length == 0) return null;

    let absolute_filename = dir + "/" + filename;

    return absolute_filename;
}

/**
 * Wrapper function to return the expanded variable value, handling the case where it doesn't exist
 * @param {Template} templ The REPORTER template object
 * @param {string} name Variable name you want to get value for
 * @returns {?string}
 */
function get_expanded_variable_value(templ, name) {
    let value = templ.GetVariableValue(name);
    if (value == null) {
        return null;
    } else {
        return templ.ExpandVariablesInString(value);
    }
}

/**
 * Wrapper function for Template.GetVariableValue that also:
 * 1. Checks that "int" and "float" variables are valid
 * 2. Updates a status object with information about any missing or invalid variables
 * 3. Creates a variable for any missing variables with the value "Missing"
 * @param {object} status Status object, used to track any missing or invalid variables
 * @param {string} name Variable  - can pass with %{name}% or just {name}
 * @param {string} [type = "string"] Variable type. Must be "string", "int", or "float".
 * @param {boolean} [required = true] Whether or not the varialbe is required (true) or optional (false)
 * @returns {*} Variable value
 * @example
 * let head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC_SCORE`, "float");
 */
function get_variable_value(status, name, type = "string", required = true) {
    let match;
    /** if the name is wrapped in % symbols then strip them off as GetVariableValue relies on name without %s */
    if ((match = name.match(/%(.*)%/))) name = match[1];
    let templ = Template.GetCurrent();
    let value = templ.GetVariableValue(name);
    if (value == null) {
        if (required) {
            status.success = false;
            status.missing.push(name);

            /* Create variable with value "Missing" so that REPORTER doesn't complain about missing
             * variables. */
            let missing_var = new Variable(
                templ,
                name,
                "Requested variable is missing",
                `Missing`,
                "String",
                false,
                true
            );
        } else {
            /* If the variable was not required, create variable with value "N/A" so that REPORTER
             * doesn't complain about missing variables, and so that user can see it is not
             * required. */
            let optional_var = new Variable(
                templ,
                name,
                "Requested variable is missing but optional so mark as N/A",
                `N/A`,
                "String",
                false,
                true
            );
        }

        return value;
    }
    if (type == "string") {
        /* For string variables, just return value without any further checks */
        return value;
    } else if (type == "int" || type == "float") {
        /* For numeric values, check they are valid integers or floats */
        let parsed;
        if (type == "int") {
            parsed = Number.parseInt(value);
        }
        if (type == "float") {
            parsed = Number.parseFloat(value);
        }
        if (Number.isNaN(parsed)) {
            status.success = false;
            status.invalid.push(name);
        }
        return parsed;
    } else {
        /* We shouldn't get here so print error and exit */
        LogError(`Unexpected argument <type = ${type}> in function get_variable_value.`);
        Exit();
    }
}

/**
 * Prints warnings about missing and/or invalid REPORTER variables that have been requested during
 * score calculations.
 * @param {object} status Contains information about missing/invalid variables, expected in the form { success: false, missing: [...], invalid: [...] }
 * @param {string} descriptor String describing what is affected
 * @example
 * let body_region_label = `knee, femur and pelvis`;
 * let status = { success: true, missing: [], invalid: [] };
 * status.success = false;
 * status.missing.push("M1_DRIVER_FEMUR_COMPRESSION_EXCEEDENCE_SCORE");
 * status.invalid.push("M1_DRIVER_KNEE_COMPRESSION_SCORE");
 * if (!status.success) {
 *     warn_about_missing_or_invalid_variables(status, `M1 DRIVER ${body_region_label} score calculation`);
 * }
 */
function warn_about_missing_or_invalid_variables(status, descriptor) {
    if (status.missing && status.missing.length > 0) {
        LogWarning(`The following variables required for the ${descriptor} are missing:`);
        for (let name of status.missing) {
            LogPrint(name);
        }
    }
    if (status.invalid && status.invalid.length > 0) {
        LogWarning(`The following variables used in the ${descriptor} have invalid values:`);
        for (let name of status.invalid) {
            LogPrint(name);
        }
    }
}

/**
 * Returns a list of models based on the value of the %MODEL_LIST% variable.
 * Expect the value of %MODEL_LIST% to be in the form:
 *
 *     "M1, M2, M3"
 *
 *  Return null if missing
 *  @returns {?string[]}
 */
function get_model_list() {
    let templ = Template.GetCurrent();
    let models_list_str = templ.GetVariableValue("MODEL_LIST");
    if (models_list_str == null) {
        return null;
    } else {
        return models_list_str.split(/,\s*/);
    }
}

/**
 * Get the parent path from a directory or file path (i.e. strip off last / or \)
 * @param {String} path
 * @returns {String} parent
 */
function get_parent(path) {
    let path_index = Math.max(path.lastIndexOf("/"), path.lastIndexOf("\\"));
    let parent = path.substring(0, path_index);
    return parent;
}

/**
 * @typedef {Object} REPORTER_JSON
 * @property {String} output_directory
 * @property {String} ref_model_tag
 * @property {String} sim_model_tag
 * @property {String} ref_file_path
 * @property {String} sim_file_path
 * @property {Object} protocol_channel_tags keys are channels value is true/false (present/missing)
 * @property {Object} he_data head excursion data
 * @property {String} head_excursion_image
 * @property {String} settings_file
 * @property {Object[]} correlations
 * @property {String} results_csv
 *
 */
