/** set up global template variable for the current template */
var templ = Template.GetCurrent();
var colour_light_blue_grey = Colour.RGB(243, 247, 251); // #F3F7FB
var colour_dark_blue_grey = Colour.RGB(100, 115, 130); //#647382

var normal_props = {
    fontSize: 8,
    fillColour: colour_light_blue_grey,
    textColour: Colour.Black(),
    fontStyle: Reporter.TEXT_NORMAL,
    fontName: "Microsoft YaHei UI",
    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
};

var bold_props = {
    fontSize: 12,
    fillColour: colour_dark_blue_grey,
    textColour: Colour.White(),
    fontStyle: Reporter.TEXT_BOLD,
    fontName: "Microsoft YaHei UI",
    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
};

var header_props = {
    fontSize: 8,
    fillColour: colour_dark_blue_grey,
    textColour: Colour.White(),
    fontStyle: Reporter.TEXT_BOLD,
    fontName: "Microsoft YaHei UI",
    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
};

/** FUNCTIONS */

function unicodeReplacer(key, value) {
    if (typeof value === "string") {
        return value.replace(/[\u007F-\uFFFF]/g, function (c) {
            return "\\u" + ("0000" + c.charCodeAt(0).toString(16)).slice(-4);
        });
    }
    return value;
}

function unicodeReviver(key, value) {
    if (typeof value === "string") {
        return value.replace(/\\u[\dA-Fa-f]{4}/g, function (match) {
            return String.fromCharCode(parseInt(match.replace("\\u", ""), 16));
        });
    }
    return value;
}

/**
 * Attempt to parse the reporter json. Throw an Error if we encounter an issue (e.g. the reporter_json_file_path does not exist)
 */
function parse_reporter_json(reporter_json_file_path = null) {
    /** if the file path is not passed in then we use the reporter_data.json file in the temporary folder
     * this is the normal case. We onlt pass in the path when testing */

    if (!reporter_json_file_path) {
        let reporter_temp = Variable.GetFromName(Template.GetCurrent(), "REPORTER_TEMP").value;
        reporter_json_file_path = `${reporter_temp}/reporter_data.json`;
    }

    /** check the path exists */
    if (File.Exists(reporter_json_file_path)) {
        let reporter_json_file = new File(reporter_json_file_path, File.READ);

        let line;
        let lines = [];

        //@ts-ignore
        while ((line = reporter_json_file.ReadLongLine()) != File.EOF) {
            lines.push(line);
        }

        reporter_json_file.Close();

        /** @type {REPORTER_JSON_DATA} */
        let reporter_json = JSON.parse(lines.join(`\n`), unicodeReviver);

        /** set the output directory based on the reporter json property*/
        set_output_file_variable(reporter_json.output_directory);

        /** create the variables for populating the results tables*/
        create_reporter_variables(reporter_json);
    } else {
        throw Error(`${reporter_json_file_path} does not exist so cannot generate template.`);
    }
}

/**
 * create reporter variables from the variables array in reporter_data.json
 * @param {REPORTER_SIMVT_JSON_DATA} reporter_json
 */
function create_reporter_variables(reporter_json) {
    let temp_var;

    if (!reporter_json.hasOwnProperty("variables")) {
        throw Error(`REPORTER data is missing the variables Object`);
    }

    /** loop through every variable and convert it */
    for (let var_name in reporter_json.variables) {
        let var_value = get_unicode_text_with_padding(reporter_json.variables[var_name]);
        let type = Number.isNaN(var_value) ? "String" : "Number";

        temp_var = new Variable(
            Template.GetCurrent(),
            var_name.toUpperCase(),
            "",
            var_value,
            Number.isNaN(var_value) ? "String" : "Number",
            false, // READ_ONLY should be false, otherwise the variable will not be written when saving a REPORT
            true // TEMPORARY
        );

        // set the format and precision for the variable if it is a number
        if (type === "Number") {
            temp_var.precision = 3;
            temp_var.format = Variable.FORMAT_GENERAL;
        }
    }
}

/** typedefs */

/**
 * @typedef {Object} REPORTER_SIMVT_JSON_DATA
 * @property {String} output_directory
 * @property {Object} variables
 * @property {String} ref_model_tag
 * @property {String} sim_model_tag
 * @property {String} ref_file_path
 * @property {String} sim_file_path
 * @property {Object} protocol_channel_tags keys are channels value is true/false (present/missing)
 * @property {Object} he_data head excursion data
 * @property {String} head_excursion_image
 * @property {String} settings_file
 * @property {Object[]} correlations
 * @property {String} results_csv
 *
 */

/** extra function dependencies */

/**
 * If relative, converts a path to an absolute path based on the current working directory.
 * @param {string} path A path to a directory or filename.
 * @param {string} label The name of the path (e.g. "KEYWORD_FILE") used for print statements.
 */
function convert_to_absolute_path(path, label) {
    if (!File.Exists(path)) {
        LogError(`In function convert_to_absolute_path: specified path for ${label} does not exist: ${path}`);
        return path;
    }
    if (File.IsAbsolute(path)) {
        /* If path is already absolute, just return it. */
        return path;
    } else {
        LogPrint(`Converting ${label} to absolute path...`);
        LogPrint(`Relative path: ${path}`);
        let current_dir = GetCurrentDirectory();
        let abs_path = `${current_dir}/${path}`;
        LogPrint(`Absolute path: ${abs_path}`);
        if (!File.Exists(abs_path)) {
            /* Trap the unexpected case where the conversion hasn't worked. */
            LogError(
                `In function convert_to_absolute_path: converted absolute path does not exist. Reverting to relative path.`
            );
            return path;
        }
        return abs_path;
    }
}

/**
 * Sets the REPORTER variables %OUTPUT_DIR%
 * The latter 2 are extracted from the settings file
 * specified keyword file.
 * @param {string} output_dir Absolute path and filename of settings file
 */
function set_output_file_variable(output_dir) {
    let templ = Template.GetCurrent();

    /* If output_dir is relative path, make it absolute path. We need to do this so that
     * any other variables that end up being derived from it (DEFAULT_DIR, RESULTS_DIR, OUTPUT_DIR,
     * etc.) are absolute paths too. When REPORTER generates an Oasys item, it sets -start_in to
     * the job file directory, so it gets confused if the relative paths we have defined are
     * relative to a different starting directory. Converting everything to absolute paths avoids
     * this problem. */
    output_dir = convert_to_absolute_path(output_dir, `%OUTPUT_DIR%`);

    let path_index = Math.max(output_dir.lastIndexOf("/"), output_dir.lastIndexOf("\\"));
    let filename = output_dir.substring(path_index + 1);
    let default_job = filename; //the name of the REPORT will be the same as the output folder by default

    /* Assign to REPORTER variables. Constructor will overwrite existing variable.
     * OUTPUT_DIR should be temporary; DEFAULT_DIR and DEFAULT_JOB are not temporary. */
    let output_dir_var = new Variable(templ, `OUTPUT_DIR`, `Output Directory`, output_dir, `Directory`, false, true);
    let default_dir_var = new Variable(
        templ,
        `DEFAULT_DIR`,
        `Reporter default directory`,
        output_dir,
        `Directory`,
        false,
        false
    );
    let default_job_var = new Variable(
        templ,
        `DEFAULT_JOB`,
        `Reporter default jobname`,
        default_job,
        `File(basename)`,
        false,
        false
    );
}

/**
 * @typedef {Object} REPORTER_JSON_DATA
 * @property {String} output_directory
 * @property {Object} variables
 * @property {String} ref_model_tag
 * @property {String} sim_model_tag
 * @property {String} ref_file_path
 * @property {String} sim_file_path
 * @property {Object} protocol_channel_tags keys are channels value is true/false (present/missing)
 * @property {Object} he_data head excursion data
 * @property {String} head_excursion_image
 * @property {String} settings_file
 * @property {Object[]} correlations
 * @property {String} results_csv
 *
 */
