// module:TRUE

// DESCRIPTION: This REPORTER Javascript duplicates the template page which contains the placeholder for storey plot(s).
//              Then, it assigns the images exported by the T/HIS post script to every placeholder in all pages.
//              Currently supports workflow to plot storey forces, storey drifts

// Main code is enclosed in IIFE so that variables declared on each script item will not be
// exposed to other script items in the Reporter template.
(() => {
    // Extract Plot/Workflow input variables
    const PLOT = set_plot_inputs();
    let templ = Template.GetCurrent();
    let workflow_name = templ.GetVariableValue("WORKFLOW");

    // Call main function to generate report
    let PLOT_VARS = PLOT.VARS.find((x) => x.workflow == workflow_name);
    generate_report(PLOT_VARS, "%DEFAULT_DIR%", "%OUTPUT_DIR%", "%IMAGES_DIR%");

    // Fix the state of temporary variables and delete those not needed anymore.
    update_variables_state();
})();

/**
 * Return PLOT CONSTANTS, that contains values of input variables specific to each workflow.
 * This allows this whole script to be reusable by multiple workflows.
 */
function set_plot_inputs() {
    /** Types of plot/workflow - workflow names on definition files */
    const WORKFLOWS = Object.freeze({
        STOREY_FORCE: "Storey Force",
        STOREY_DRIFT: "Storey Drift"
    });

    /** Object holder for variables specific to plotting STOREY FORCES */
    const STOREY_FORCE_VARS = {
        workflow: WORKFLOWS.STOREY_FORCE,
        reporter_img: {
            d3plot_view_prfx: "storey_force_S_PTS",
            t_his_graph_prfx: "storey_force_G"
        }
    };

    /** Object holder for variables specific to plotting STOREY DRIFTS */
    const STOREY_DRIFT_VARS = {
        workflow: WORKFLOWS.STOREY_DRIFT,
        reporter_img: {
            d3plot_view_prfx: "storey_drift_N",
            t_his_graph_prfx: "storey_drift_D"
        }
    };

    return {
        /** Types of plot/workflow */
        WORKFLOWS: WORKFLOWS,
        /** Set of input variables specific to each plot */
        VARS: [STOREY_DRIFT_VARS, STOREY_FORCE_VARS]
    };
}

/**
 * Main script to generate Reporter template, called in Reporter Script Item.
 * This script duplicates the template pages, modifies the placeholder labels and the input files
 * so that it matches the image files generated previously by the T/HIS and D3Plot scripts.
 * @param {any} PLOT_VARS - Object of inputs specific to each PLOT/WORKFLOW
 * @param {String} default_dir_ref - Variable string for default dir (syntax: %VAR%)
 * @param {String} output_dir_ref - Variable string for output dir (syntax: %VAR%)
 * @param {String} images_dir_ref - Variable string for images dir (syntax: %VAR%)
 */
function generate_report(PLOT_VARS, default_dir_ref, output_dir_ref, images_dir_ref) {
    // Get template object
    let templ = Template.GetCurrent();

    //#region Main Script
    // Duplicate pages and assign plot images to placeholders in each page
    //  Page 0 => summary page
    //  Page 1 => page to duplicate
    let page_num = 1; // page number to duplicate (zero-based)
    let gid = 1; // graph id is one-based, consistent with T/HIS and workflow data
    let num_graphs_per_page = Number.parseInt(templ.GetVariableValue("NUM_GRAPHS_PER_PAGE")); // number of pairs of plots to show in the page
    let num_graphs = Number.parseInt(templ.GetVariableValue("NUM_GRAPHS"));
    let num_pages_to_gen = Math.ceil(num_graphs / num_graphs_per_page); // number of pages to generate
    let templ_curr_num_pages = templ.pages; // current number of pages in the template

    // Template item prefixes
    let section_title_pref = "EntityTag";
    let section_subtitle_pref = "CompTag";
    let model_view_pref = "ModelView";
    let graph_view_pref = "Graph";

    for (let i = 0; i < num_pages_to_gen; i++) {
        // Deactivate all items (placeholders) in master page first
        let page = templ.GetPage(page_num);
        let items = page.GetAllItems();
        items.forEach((p) => (p.active = false));

        // Set values text/label items to blank
        // -> When excess page items remain at the end of run, this ensures they are not referring to any image file, which might lead to run errors.
        for (let j = 0; j < num_graphs_per_page; j++) {
            update_item(`${section_title_pref}${j + 1}`, "text", "", items);
            update_item(`${section_subtitle_pref}${j + 1}`, "text", "", items);
        }

        // Update textbox and placeholders of graphs in the page
        for (let j = 0; j < num_graphs_per_page; j++) {
            if (gid > num_graphs) break; // if graph setup does not exist, exit

            /** @type {Item[]} */
            let updated_items = [];

            // Update textboxes/labels
            let graph_title = templ.GetVariableValue(`G${String(gid).padStart(3, "0")}_TITLE`);
            let graph_subtitle = templ.GetVariableValue(`G${String(gid).padStart(3, "0")}_SUBTITLE`);

            updated_items.push(update_item(`${section_title_pref}${j + 1}`, "text", graph_title, items));
            updated_items.push(update_item(`${section_subtitle_pref}${j + 1}`, "text", graph_subtitle, items));

            // Replace linked file in placeholder with highlighted D3PLOT graphic image
            let d3plot_img_file = `${default_dir_ref}/${output_dir_ref}/${images_dir_ref}/${
                PLOT_VARS.reporter_img.d3plot_view_prfx
            }${String(gid).padStart(3, "0")}.png`;
            updated_items.push(update_item(`${model_view_pref}${j + 1}`, "file", d3plot_img_file, items));

            // Replace linked file in placeholder with T/HIS plot image
            let this_img_file = `${default_dir_ref}/${output_dir_ref}/${images_dir_ref}/${
                PLOT_VARS.reporter_img.t_his_graph_prfx
            }${String(gid).padStart(3, "0")}.png`;
            updated_items.push(update_item(`${graph_view_pref}${j + 1}`, "file", this_img_file, items));

            // Activate all updated items
            updated_items.forEach((item) => (item.active = true));

            gid += 1; // Go to next graph
        }

        // Duplicate master page
        if (i < num_pages_to_gen - 1) {
            page_num += 1;
            if (page_num + 1 > templ_curr_num_pages) page.Duplicate();
        }
    }
    //#endregion

    /**
     * Update a property value of a Reporter template item.
     * Item is searched by input name and item property and value should be specified to update.
     * This supports updating multiple items by seaching providing substring in item_name instead, common to all items to be updated.
     * @param {String} item_name - Name (of part of name [i.e. identifiable prefix]) of template item that will be updated
     * @param {String} item_prop - Property of item to be updated (e.g. text for item of type ITEM.TEXT, file for item of type ITEM.IMAGE_FILE)
     * @param {String} item_prop_val - Value of property to update
     * @param {Item[]} templ_items - Array of template items, to search item from
     */
    function update_item(item_name, item_prop, item_prop_val, templ_items) {
        let item = templ_items.find((i) => i.name.includes(item_name));
        item[item_prop] = item_prop_val;

        return item;
    }
}

/**
 * Sets state of variables to temporary or not.
 *
 * Running the Reporter Library Program 'ReadVariablesFile' sets all variables to 'Temporary' except those of type 'File' or 'Directory'.
 * Thus, this needs to be ran after the Library Program to fix the state of variables.
 *
 * Some set of temporary variables are deleted while some are retained for user record.
 */
function update_variables_state() {
    let templ = Template.GetCurrent();

    let variables = Variable.GetAll(templ);

    LogPrint(`Setting all "not read only" variables as permanent.`);
    variables
        .filter((v) => v.readonly == false)
        .forEach((v) => {
            if (!(v.type == "Directory" || v.type.startsWith("File"))) {
                v.temporary = false;

                // @ts-ignore
                if (!isNaN(v.value) && !isNaN(parseFloat(v.value))) {
                    v.type = "Number";
                } else {
                    v.type = "String";
                }
            }
        });

    LogPrint(`Setting "Graph Title and Subtitle" variables as temporary.`);
    variables
        .filter((v) => v.name.startsWith("G") && v.name.endsWith("_TITLE"))
        .forEach((v) => {
            v.temporary = true;
        });
    variables
        .filter((v) => v.name.startsWith("G") && v.name.endsWith("_SUBTITLE"))
        .forEach((v) => {
            v.temporary = true;
        });
    LogPrint(`Deleting "Graph Title and Subtitle" variables.`);
    templ.DeleteTemporaryVariables();

    // Model paths are set to temporary so it can be deleted when the template is ran again, but with less models selected.
    // However, they are not deleted for user record.
    LogPrint(`Setting "Model path" variables as temporary.`);
    // Refresh variables list because we just deleted some temporary variables
    variables = Variable.GetAll(templ);
    variables
        .filter((v) => v.name.startsWith("MODEL_PATH_"))
        .forEach((v) => {
            v.temporary = true;
        });
}
