/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
neck_nij_values();
neck_nij_probability_score();
neck_overall_score();

/**
 * Create variables for the maximum NIJ values for the driver and front passenger.
 */
function neck_nij_values() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `neck_nij`;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    for (let m of models) {
        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded */

        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Neck NIJ
             *
             * We find the maximum value of the four components of Neck NIJ to find the net result for it. In doing so we find the maximum of tension extension, tension, flexion, compression extension , compression flexion */

            let tension_extension_driver;
            let tension_flexion_driver;
            let compression_extension_driver;
            let compression_flexion_driver;

            let tension_extension_pass;
            let tension_flexion_pass;
            let compression_extension_pass;
            let compression_flexion_pass;

            switch (occ) {
                case "DRIVER":
                    tension_extension_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_TENSION-EXTENSION_VALUE`,
                        "float",
                        false
                    );
                    tension_flexion_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_TENSION-FLEXION_VALUE`,
                        "float",
                        false
                    );
                    compression_extension_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_COMPRESSION-EXTENSION_VALUE`,
                        "float",
                        false
                    );
                    compression_flexion_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_COMPRESSION-FLEXION_VALUE`,
                        "float",
                        false
                    );

                    break;
                case "FRONT_PASSENGER":
                    tension_extension_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_TENSION-EXTENSION_VALUE`,
                        "float",
                        false
                    );
                    tension_flexion_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_TENSION-FLEXION_VALUE`,
                        "float",
                        false
                    );
                    compression_extension_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_COMPRESSION-EXTENSION_VALUE`,
                        "float",
                        false
                    );
                    compression_flexion_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_COMPRESSION-FLEXION_VALUE`,
                        "float",
                        false
                    );
                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let neck_nij_total_force_driver = 0;
            let neck_nij_driver = 0;

            let neck_nij_total_force_passenger = 0;
            let neck_nij_passenger = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                switch (occ) {
                    case "DRIVER":
                        neck_nij_driver = Math.max(
                            tension_extension_driver,
                            Math.max(tension_flexion_driver, compression_extension_driver)
                        );
                        neck_nij_total_force_driver = Math.max(neck_nij_driver, compression_flexion_driver);
                        LogPrint(`${m} ${occ} ${body_region_label} force = ${neck_nij_total_force_driver}`);
                        break;
                    case "FRONT_PASSENGER":
                        neck_nij_passenger = Math.max(
                            tension_extension_pass,
                            Math.max(tension_flexion_pass, compression_extension_pass)
                        );
                        neck_nij_total_force_passenger = Math.max(neck_nij_passenger, compression_flexion_pass);
                        LogPrint(`${m} ${occ} ${body_region_label} force = ${neck_nij_total_force_passenger}`);
                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            if (occ == "DRIVER") {
                let pelvis_force_var = new Variable(
                    templ,
                    `${m}_${occ}_NECK_NIJ_TOTAL_VALUE`,
                    `Maximum ${body_region_label} force`,
                    neck_nij_total_force_driver.toString(),
                    "String",
                    false,
                    true
                );
            }
            if ((occ = "FRONT_PASSENGER")) {
                let pelvis_force_score_var = new Variable(
                    templ,
                    `${m}_${occ}_NECK_NIJ_TOTAL_VALUE`,
                    `Maximum ${body_region_label} force`,
                    neck_nij_total_force_passenger.toString(),
                    "String",
                    false,
                    true
                );
            }
        }
    }
}

/**
 * Create variables for the maximum NIJ probability scores for the driver and front passenger.
 */
function neck_nij_probability_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `neck_nij`;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    for (let m of models) {
        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded */

        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Neck NIJ
             *
             * We find the maximum value of the four components of Neck NIJ to find the net result for it. In doing so we find the maximum of tension extension, tension, flexion, compression extension , compression flexion */

            let tension_extension_driver;
            let tension_flexion_driver;
            let compression_extension_driver;
            let compression_flexion_driver;

            let tension_extension_pass;
            let tension_flexion_pass;
            let compression_extension_pass;
            let compression_flexion_pass;

            switch (occ) {
                case "DRIVER":
                    tension_extension_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_TENSION-EXTENSION_SCORE`,
                        "float",
                        false
                    );
                    tension_flexion_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_TENSION-FLEXION_SCORE`,
                        "float",
                        false
                    );
                    compression_extension_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_COMPRESSION-EXTENSION_SCORE`,
                        "float",
                        false
                    );
                    compression_flexion_driver = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_COMPRESSION-FLEXION_SCORE`,
                        "float",
                        false
                    );

                    break;
                case "FRONT_PASSENGER":
                    tension_extension_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_TENSION-EXTENSION_SCORE`,
                        "float",
                        false
                    );
                    tension_flexion_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_TENSION-FLEXION_SCORE`,
                        "float",
                        false
                    );
                    compression_extension_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_COMPRESSION-EXTENSION_SCORE`,
                        "float",
                        false
                    );
                    compression_flexion_pass = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_NIJ_PASSENGER_COMPRESSION-FLEXION_SCORE`,
                        "float",
                        false
                    );
                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let neck_nij_total_prob_driver = 0;
            let neck_nij_driver = 0;

            let neck_nij_total_prob_passenger = 0;
            let neck_nij_passenger = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                switch (occ) {
                    case "DRIVER":
                        neck_nij_driver = Math.max(
                            tension_extension_driver,
                            Math.max(tension_flexion_driver, compression_extension_driver)
                        );
                        neck_nij_total_prob_driver = Math.max(neck_nij_driver, compression_flexion_driver);
                        LogPrint(`${m} ${occ} ${body_region_label} probability = ${neck_nij_total_prob_driver}`);
                        break;
                    case "FRONT_PASSENGER":
                        neck_nij_passenger = Math.max(
                            tension_extension_pass,
                            Math.max(tension_flexion_pass, compression_extension_pass)
                        );
                        neck_nij_total_prob_passenger = Math.max(neck_nij_passenger, compression_flexion_pass);
                        LogPrint(`${m} ${occ} ${body_region_label} probability = ${neck_nij_total_prob_passenger}`);
                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            if (occ == "DRIVER") {
                let pelvis_force_var = new Variable(
                    templ,
                    `${m}_${occ}_NECK_NIJ_TOTAL_SCORE`,
                    `Maximum ${body_region_label} force`,
                    neck_nij_total_prob_driver.toString(),
                    "String",
                    false,
                    true
                );
            }
            if ((occ = "FRONT_PASSENGER")) {
                let pelvis_force_score_var = new Variable(
                    templ,
                    `${m}_${occ}_NECK_NIJ_TOTAL_SCORE`,
                    `Maximum ${body_region_label} force`,
                    neck_nij_total_prob_passenger.toString(),
                    "String",
                    false,
                    true
                );
            }
        }
    }
}

/**
 * Creates variables for the overall neck injury probability scores for the driver and front passenger.
 */
function neck_overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} neck injury probability...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            switch (occ) {
                case "DRIVER": {
                    let neck_compression_exceedence_val = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_COMPRESSION_EXCEEDENCE_MAX_VALUE`,
                        "float"
                    );
                    let neck_nij_val = get_variable_value(status, `${m}_${occ}_NECK_NIJ_TOTAL_VALUE`, "float");
                    let neck_tension_exceedence_val = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_TENSION_EXCEEDENCE_MAX_VALUE`,
                        "float"
                    );

                    let neck_compression_exceedence_prob = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_COMPRESSION_EXCEEDENCE_SCORE`,
                        "float"
                    );
                    let neck_nij_prob = get_variable_value(status, `${m}_${occ}_NECK_NIJ_TOTAL_SCORE`, "float");
                    let neck_tension_exceedence_prob = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_TENSION_EXCEEDENCE_SCORE`,
                        "float"
                    );

                    /* If we have all the required variables, calculate the final scores */
                    if (status.success) {
                        let neck_injury_val = Math.max(
                            neck_compression_exceedence_val,
                            Math.max(neck_nij_val, neck_tension_exceedence_val)
                        );
                        let neck_injury_prob = Math.max(
                            neck_compression_exceedence_prob,
                            Math.max(neck_nij_prob, neck_tension_exceedence_prob)
                        );

                        let total_neck_force_val = new Variable(
                            templ,
                            `${m}_${occ}_TOTAL_NECK_INJURY_FORCE_VAL`,
                            `${occ} TOTAL FORCES ON NECK VALUE`,
                            neck_injury_val.toString(),
                            "String",
                            false,
                            true
                        );

                        let total_neck_prob = new Variable(
                            templ,
                            `${m}_${occ}_NECK_INJURY_PROB`,
                            `${occ} neck force max value`,
                            neck_injury_prob.toString(),
                            "String",
                            false,
                            true
                        );
                    } else {
                        warn_about_missing_or_invalid_variables(
                            status,
                            `${m} ${occ} neck value and probability calculation`
                        );
                    }

                    break;
                }

                case "FRONT_PASSENGER": {
                    let neck_compression_exceedence_val = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_COMPRESSION_EXCEEDENCE_PASSENGER_MAX_VALUE`,
                        "float"
                    );
                    let neck_nij_val = get_variable_value(status, `${m}_${occ}_NECK_NIJ_TOTAL_VALUE`, "float");
                    let neck_tension_exceedence_val = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_TENSION_EXCEEDENCE_PASSENGER_MAX_VALUE`,
                        "float"
                    );

                    let neck_compression_exceedence_prob = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_COMPRESSION_EXCEEDENCE_PASSENGER_SCORE`,
                        "float"
                    );
                    let neck_nij_prob = get_variable_value(status, `${m}_${occ}_NECK_NIJ_TOTAL_SCORE`, "float");
                    let neck_tension_exceedence_prob = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_TENSION_EXCEEDENCE_PASSENGER_SCORE`,
                        "float"
                    );

                    /* If we have all the required variables, calculate the final scores */
                    if (status.success) {
                        let neck_injury_val = Math.max(
                            neck_compression_exceedence_val,
                            Math.max(neck_nij_val, neck_tension_exceedence_val)
                        );
                        let neck_injury_prob = Math.max(
                            neck_compression_exceedence_prob,
                            Math.max(neck_nij_prob, neck_tension_exceedence_prob)
                        );

                        let total_neck_force_val = new Variable(
                            templ,
                            `${m}_${occ}_TOTAL_NECK_INJURY_FORCE_VAL`,
                            `${occ} TOTAL FORCES ON NECK VALUE`,
                            neck_injury_val.toString(),
                            "String",
                            false,
                            true
                        );

                        let total_neck_prob = new Variable(
                            templ,
                            `${m}_${occ}_NECK_INJURY_PROB`,
                            `${occ} neck force max value`,
                            neck_injury_prob.toString(),
                            "String",
                            false,
                            true
                        );
                    } else {
                        warn_about_missing_or_invalid_variables(
                            status,
                            `${m} ${occ} neck value and probability calculation`
                        );
                    }

                    break;
                }

                default:
                    LogError(`Unexpected occupant type "${occ}".`);
                    Exit();
            }
        }
    }
}
