/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest and abdomen score calculation for the specified models
 */
function chest_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `chest`;

    let occupants = ["DRIVER"];
    for (let m of models) {
        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded */
        let capping_limit_explanation = "";
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Chest score and modifiers */

            /* Driver */
            let incorrect_airbag_mod = get_variable_value(
                status,
                `${m}_CHEST_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`,
                "int"
            );

            let comp_scores = get_multiple_variable_values(
                status,
                [
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_01_SCORE`,
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_02_SCORE`,
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_03_SCORE`
                ],
                "float",
                true
            );

            let vc_ratings = get_multiple_variable_values(
                status,
                [
                    `${m}_${occ}_CHEST_VISCOUS_CRITERION_UPPER_MAX_RATING`,
                    `${m}_${occ}_CHEST_VISCOUS_CRITERION_MIDDLE_MAX_RATING`,
                    `${m}_${occ}_CHEST_VISCOUS_CRITERION_LOWER_MAX_RATING`
                ],
                "string",
                true
            );

            let shoulder_ratings = get_multiple_variable_values(
                status,
                [
                    `${m}_${occ}_SHOULDER_LATERAL_FORCES_LEFT_MAX_RATING`,
                    `${m}_${occ}_SHOULDER_LATERAL_FORCES_RIGHT_MAX_RATING`
                ],
                "string",
                true
            );

            let capping_limits = get_multiple_variable_values(
                status,
                [
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_01_CAPPING_LIMIT`,
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_02_CAPPING_LIMIT`,
                    `${m}_${occ}_CHEST_COMPRESSION_MAX_03_CAPPING_LIMIT`
                ],
                "string",
                true
            );

            let chest_vc_mod = return_pass_or_fail(Object.values(vc_ratings));

            new Variable(
                templ,
                `${m}_${occ}_CHEST_VISCOUS_CRITERION_MODIFIER`,
                "Chest VC Modifier",
                chest_vc_mod.toString(),
                "String",
                false,
                true
            );

            let shoulder_mod = return_pass_or_fail(Object.values(shoulder_ratings));

            new Variable(
                templ,
                `${m}_${occ}_SHOULDER_MODIFIER`,
                "Shoulder Modifier",
                shoulder_mod.toString(),
                "String",
                false,
                true
            );

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let chest_score_before_mods = 0;
            let chest_score = 0;
            /* Capping limits start as exceeded and will remain so if any variables were missing or invalid */
            let chest_capping_limit = "*";

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Check capping limits. If any individual limits are exceeded, an asterisk is
                 * displayed for all parent body regions. */
                if (return_capping_limit(Object.values(capping_limits)) == "") {
                    chest_capping_limit = "";
                } else {
                    capping_limit_explanation = `*Capping limit exceeded`;
                }

                /* Calculate the final score for the chest region */
                chest_score_before_mods = Math.min(...Object.values(comp_scores));

                chest_score = Math.max(0, chest_score_before_mods + incorrect_airbag_mod);

                /* Check if the chest viscous criterion or shoulder criterion have failed and if so set the score to 0 */
                if (chest_vc_mod == "Fail") {
                    if (capping_limit_explanation == "") {
                        capping_limit_explanation += `Chest viscous criterion limit exceeded`;
                    } else {
                        capping_limit_explanation += `, Chest viscous criterion limit exceeded`;
                    }

                    chest_score = 0;
                }
                if (shoulder_mod == "Fail") {
                    if (capping_limit_explanation == "") {
                        capping_limit_explanation += `Shoulder lateral force limit exceeded`;
                    } else {
                        capping_limit_explanation += `, Shoulder lateral force limit exceeded`;
                    }

                    chest_score = 0;
                }

                LogPrint(`${m} ${occ} chest score = ${chest_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            new Variable(
                templ,
                `${m}_${occ}_CHEST_COMPRESSION_SCORE`,
                "Minimum of left and right upper/lower compression scores",
                chest_score_before_mods.toString(),
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_${occ}_CHEST_FINAL_SCORE`,
                "Final chest score",
                chest_score.toString(),
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_${occ}_CHEST_CAPPING_LIMIT`,
                `Chest capping limit`,
                chest_capping_limit,
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_CHEST_CAPPING_LIMIT_EXPLANATION`,
                `Capping limit explanation`,
                capping_limit_explanation,
                "String",
                false,
                true
            );
        }
    }
}
