/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
overall_score();

/**
 * Performs the overall score calculation for the specified models
 */
function overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE"];

    /* The following objects will be populated with values for each occupant */
    let head_score = {};
    let neck_score = {};
    let chest_score = {};
    let upper_leg_score = {};
    let occ_score = {};
    let occ_capping_limit = {};
    let final_driver_score = 0;
    let final_front_passenger_score = 0;
    let final_Rear_driver_side_score = 0;
    let excessive_mvmt_dummy_head;

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} overall score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Body region final scores */
            head_score[occ] = get_variable_value(status, `${m}_${occ}_HEAD_FINAL_SCORE`, "float");
            neck_score[occ] = get_variable_value(status, `${m}_${occ}_NECK_FINAL_SCORE`, "float");
            chest_score[occ] = get_variable_value(status, `${m}_${occ}_CHEST_FINAL_SCORE`, "float");
            upper_leg_score[occ] = get_variable_value(status, `${m}_${occ}_UPPER_LEG_FINAL_SCORE`, "float");

            /* Rear passenger modifier */
            if (occ == "REAR_DRIVER_SIDE") {
                excessive_mvmt_dummy_head = get_variable_value(
                    status,
                    `${m}_EXCESSIVE_MVMT_DUMMY_HEAD_MODIFIER`,
                    "float"
                );
            }

            /* Body region capping limits */
            let head_capping_limit = get_variable_value(status, `${m}_${occ}_HEAD_CAPPING_LIMIT`, "string");
            let neck_capping_limit = get_variable_value(status, `${m}_${occ}_NECK_CAPPING_LIMIT`, "string");
            let chest_capping_limit = get_variable_value(status, `${m}_${occ}_CHEST_CAPPING_LIMIT`, "string");

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            occ_score[occ] = 0;
            occ_capping_limit[occ] = "*";

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Overall occupant score is sum of body region scores. */
                occ_score[occ] = head_score[occ] + neck_score[occ] + chest_score[occ] + upper_leg_score[occ];
                LogPrint(`${m} ${occ} overall score = ${occ_score[occ]}`);

                /* Only remove the overall occupant capping limit if none of the body region
                 * capping limits were exceeded. */

                if (occ == "DRIVER") {
                    final_driver_score = occ_score[occ];
                    LogPrint("the final driver score" + final_driver_score);

                    // for the driver, there are capping limits for the head, neck and chest
                    if (head_capping_limit == "" && neck_capping_limit == "" && chest_capping_limit == "") {
                        occ_capping_limit[occ] = "";
                        LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                    } else {
                        LogPrint(`Capping limit applied to ${m} ${occ}.`);
                    }
                } else if (occ == "FRONT_PASSENGER") {
                    final_front_passenger_score = occ_score[occ];
                    LogPrint("the final front passenger score" + final_front_passenger_score);

                    // for the front passenger, there are capping limits for the head, neck and chest
                    if (head_capping_limit == "" && neck_capping_limit == "" && chest_capping_limit == "") {
                        occ_capping_limit[occ] = "";
                        LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                    } else {
                        LogPrint(`Capping limit applied to ${m} ${occ}.`);
                    }
                } else if ((occ = "REAR_DRIVER_SIDE")) {
                    occ_score[occ] = occ_score[occ] + excessive_mvmt_dummy_head;
                    final_Rear_driver_side_score = occ_score[occ];
                    LogPrint("the final rear driver side score" + final_Rear_driver_side_score);

                    // for the rear passenger, only the head and chest body regions have capping limits
                    if (head_capping_limit == "" && chest_capping_limit == "") {
                        occ_capping_limit[occ] = "";
                        LogPrint(`Capping limit not applied to ${m} ${occ}.`);
                    } else {
                        LogPrint(`Capping limit applied to ${m} ${occ}.`);
                    }
                } else {
                    LogError(`Unexpected occupant type "${occ}" in overall score calculation.`);
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} overall score calculation`);
            }

            new Variable(
                templ,
                `${m}_${occ}_OVERALL_SCORE`,
                `Overall ${occ} score`,
                occ_score[occ].toString(),
                "String",
                false,
                true
            );
        }

        LogPrint(`Retrieving door opening modifier...`);
        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };
        let door_opening_mod = get_variable_value(status, `${m}_DOOR_OPENING_MODIFIER`, "int");
        let fuel_leakage_mod = get_variable_value(status, `${m}_FUEL_LEAKAGE_MODIFIER`, "int");
        if (!status.success) {
            warn_about_missing_or_invalid_variables(status, `${m} door opening modifier`);
            door_opening_mod = 0;
            fuel_leakage_mod = 0;
        }
        let front_seat_overall_score = 0.0;
        let overall_score = 0.0;
        let overall_capping_limit = "";

        /* Overall capping limit applied if either occupant's capping limit exceeded */
        /* if capping limit is applied, the overall_score will remain at the initialised value of zero*/
        if (
            return_capping_limit([
                occ_capping_limit.DRIVER,
                occ_capping_limit.FRONT_PASSENGER,
                occ_capping_limit.REAR_DRIVER_SIDE
            ]) == "*"
        ) {
            overall_capping_limit = "*";
            LogPrint(`Overall capping limit applied. ${m} overall score = ${overall_score}`);
        } else {
            /* the overall score is the average of the front and rear occupants, then with the
             *door opening and fuel leakage modifiers added (where the front occupant score is
             *taken as the minimum of that of the driver and the front passenger).
             The overall score is scaled by 13/16 to convert to a score out of 13
             */
            front_seat_overall_score =
                Math.min(head_score.DRIVER, head_score.FRONT_PASSENGER) +
                Math.min(neck_score.DRIVER, neck_score.FRONT_PASSENGER) +
                Math.min(chest_score.DRIVER, chest_score.FRONT_PASSENGER) +
                Math.min(upper_leg_score.DRIVER, upper_leg_score.FRONT_PASSENGER);

            overall_score = Math.max(
                (13 / 16) *
                    (0.5 * (front_seat_overall_score + final_Rear_driver_side_score) +
                        door_opening_mod +
                        fuel_leakage_mod),
                0
            );

            LogPrint(`${m} overall score = ${overall_score}`);
            new Variable(
                templ,
                `${m}_FRONT_OVERALL_SCORE`,
                `front seat score`,
                front_seat_overall_score.toString(),
                "String",
                false,
                true
            );
        }

        new Variable(templ, `${m}_OVERALL_SCORE`, `Overall score`, overall_score.toString(), "String", false, true);

        new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT`,
            `Overall capping limit`,
            overall_capping_limit,
            "String",
            false,
            true
        );
        let capping_limit_explanation = `*Capping limit exceeded`;
        if (overall_capping_limit == "") {
            capping_limit_explanation = "";
        }
        new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
