/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
neck_score();

/**
 * Performs the neck score calculation for the specified models
 */
function neck_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `neck`;

    let occupants = ["DRIVER", "REAR_PASSENGER_SIDE"];

    /* Weighting factors */
    let driver_neck_wt_factor = 0.231;
    let passenger_neck_wt_factor = 0.2;

    new Variable(
        templ,
        "DRIVER_NECK_WEIGHTING_FACTOR",
        "Driver neck weighting factor",
        driver_neck_wt_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "REAR_PASSENGER_SIDE_NECK_WEIGHTING_FACTOR",
        "Rear passenger side neck weighting factor",
        passenger_neck_wt_factor.toString(),
        "String",
        false,
        true
    );

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Neck score and modifiers */

            /* Shared */
            let neck_tension_score;
            let neck_shear_score;
            let neck_extension_score;

            /* Passenger */
            let secondary_collision_modifier;

            switch (occ) {
                case "DRIVER":
                    neck_tension_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_TENSION_EXCEEDENCE_SCORE`,
                        "float"
                    );
                    neck_shear_score = get_variable_value(status, `${m}_${occ}_NECK_SHEAR_EXCEEDENCE_SCORE`, "float");
                    neck_extension_score = get_variable_value(status, `${m}_${occ}_NECK_EXTENSION_SCORE`, "float");

                    break;

                case "REAR_PASSENGER_SIDE":
                    neck_tension_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_AXIAL_REAR_PASSENGER_SCORE`,
                        "float"
                    );
                    neck_shear_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_SHEAR_REAR_PASSENGER_SCORE`,
                        "float"
                    );
                    neck_extension_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_EXTENSION_REAR_PASSENGER_SCORE`,
                        "float"
                    );
                    secondary_collision_modifier = get_variable_value(
                        status,
                        `${m}_${occ}_SECONDARY_COLLISION_MODIFIER`,
                        "string"
                    );
                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let neck_tension_score_weighted = 0;
            let neck_shear_score_weighted = 0;
            let neck_extension_score_weighted = 0;

            let neck_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Neck score is:
                 * For the driver, ( minimum of tension, shear, and extension bending components ) * weighting factor
                 * For the rear passenger,
                 *      if secondary collision = "Yes", then score is ( minimum of tension, shear, and extension bending components ) * weighting factor
                 *      if secondary collision = "No", then score is tensile load * weighting factor
                 */

                if (occ == "DRIVER") {
                    neck_tension_score_weighted = neck_tension_score * driver_neck_wt_factor;
                    neck_shear_score_weighted = neck_shear_score * driver_neck_wt_factor;
                    neck_extension_score_weighted = neck_extension_score * driver_neck_wt_factor;

                    neck_score = Math.min(
                        neck_tension_score_weighted,
                        neck_shear_score_weighted,
                        neck_extension_score_weighted
                    );
                } else {
                    neck_tension_score_weighted = neck_tension_score * passenger_neck_wt_factor;
                    neck_shear_score_weighted = neck_shear_score * passenger_neck_wt_factor;
                    neck_extension_score_weighted = neck_extension_score * passenger_neck_wt_factor;

                    if (secondary_collision_modifier == "Yes") {
                        neck_score = Math.min(
                            neck_tension_score_weighted,
                            neck_shear_score_weighted,
                            neck_extension_score_weighted
                        );
                    } else if (secondary_collision_modifier == "No") {
                        neck_score = neck_tension_score_weighted;
                    }
                }

                LogPrint(`${m} ${occ} neck score = ${neck_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Weighted scores */
            new Variable(
                templ,
                `${m}_${occ}_NECK_TENSION_SCORE_WEIGHTED`,
                "Neck tension score weighted",
                neck_tension_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_NECK_SHEAR_SCORE_WEIGHTED`,
                "Neck shear score weighted",
                neck_shear_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_NECK_EXTENSION_SCORE_WEIGHTED`,
                "Neck extension score weighted",
                neck_extension_score_weighted.toString(),
                "String",
                false,
                true
            );

            /* Overall scores */
            new Variable(
                templ,
                `${m}_${occ}_NECK_FINAL_SCORE`,
                "Final neck score",
                neck_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
