/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
lower_leg();

/**
 * Performs the lower leg, foot and ankle score calculation for the specified models
 */
function lower_leg() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `lower leg`;

    let occupants = ["DRIVER", "REAR_PASSENGER_SIDE"];
    let sides = ["LEFT", "RIGHT"];

    /* Weighting factors */
    let driver_lower_leg_wt_factor = 0.923;
    let passenger_lower_leg_wt_factor = 0.4;

    new Variable(
        templ,
        "DRIVER_LOWER_LEG_WEIGHTING_FACTOR",
        "Driver lower leg weighting factor",
        driver_lower_leg_wt_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "REAR_PASSENGER_SIDE_LOWER_LEG_WEIGHTING_FACTOR",
        "Rear passenger side lower leg weighting factor",
        passenger_lower_leg_wt_factor.toString(),
        "String",
        false,
        true
    );

    for (let m of models) {
        for (let occ of occupants) {
            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Lower leg score and modifiers */

            /* Shared */
            let left_femur_comp_score;
            let right_femur_comp_score;

            /* Driver */
            let left_tibia_index_upper_score;
            let left_tibia_index_lower_score;
            let right_tibia_index_upper_score;
            let right_tibia_index_lower_score;

            let brake_pedal_vertical_score;
            let brake_pedal_fore_aft_score;

            let left_tibia_comp_upper_score;
            let left_tibia_comp_lower_score;
            let right_tibia_comp_upper_score;
            let right_tibia_comp_lower_score;

            switch (occ) {
                case "DRIVER":
                    left_femur_comp_score = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_FEMUR_COMPRESSION_SCORE`,
                        "float",
                        true
                    );

                    right_femur_comp_score = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_FEMUR_COMPRESSION_SCORE`,
                        "float",
                        true
                    );

                    left_tibia_index_upper_score = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_TIBIA_INDEX_UPPER_MAX_SCORE`,
                        "float",
                        true
                    );

                    right_tibia_index_upper_score = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_TIBIA_INDEX_UPPER_MAX_SCORE`,
                        "float",
                        true
                    );

                    left_tibia_index_lower_score = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_TIBIA_INDEX_LOWER_MAX_SCORE`,
                        "float",
                        true
                    );

                    right_tibia_index_lower_score = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_TIBIA_INDEX_LOWER_MAX_SCORE`,
                        "float",
                        true
                    );

                    left_tibia_comp_upper_score = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_TIBIA_COMPRESSION_UPPER_MAX_SCORE`,
                        "float",
                        true
                    );

                    right_tibia_comp_upper_score = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_TIBIA_COMPRESSION_UPPER_MAX_SCORE`,
                        "float",
                        true
                    );

                    left_tibia_comp_lower_score = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_TIBIA_COMPRESSION_LOWER_MAX_SCORE`,
                        "float",
                        true
                    );

                    right_tibia_comp_lower_score = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_TIBIA_COMPRESSION_LOWER_MAX_SCORE`,
                        "float",
                        true
                    );

                    brake_pedal_vertical_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_VERTICAL_INTRUSION_SCORE`,
                        "float"
                    );
                    brake_pedal_fore_aft_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_BRAKE_PEDAL_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );

                    break;

                case "REAR_PASSENGER_SIDE":
                    left_femur_comp_score = get_variable_value(
                        status,
                        `${m}_${occ}_LEFT_FEMUR_COMPRESSION_PASSENGER_SCORE`,
                        "float",
                        true
                    );

                    right_femur_comp_score = get_variable_value(
                        status,
                        `${m}_${occ}_RIGHT_FEMUR_COMPRESSION_PASSENGER_SCORE`,
                        "float",
                        true
                    );

                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* If we have all the required variables, calculate the final scores */
            let left_femur_comp_score_weighted = 0;
            let right_femur_comp_score_weighted = 0;

            let left_tibia_index_upper_score_weighted = 0;
            let right_tibia_index_upper_score_weighted = 0;
            let left_tibia_index_lower_score_weighted = 0;
            let right_tibia_index_lower_score_weighted = 0;

            let left_tibia_comp_upper_score_weighted = 0;
            let right_tibia_comp_upper_score_weighted = 0;
            let left_tibia_comp_lower_score_weighted = 0;
            let right_tibia_comp_lower_score_weighted = 0;

            let brake_pedal_vertical_score_weighted = 0;
            let brake_pedal_fore_aft_score_weighted = 0;

            let final_femur_score = 0;
            let final_tibia_index_score = 0;
            let final_tibia_comp_score = 0;

            let lower_leg_final_score = 0;

            if (status.success) {
                /* Score calculation is different for different occupants */
                /* "Lower-leg score" is:
                 * For the driver, {
                 *                    (sum of [minimum of the tibia index] and [minimum of the femur loads])
                 *                     minus the brake pedal scores and tibia compression modifier scores
                 *                 } *  weighting factor
                 * For the rear passenger, (minimum of the femur compression loads) * weighting factor
                 */

                LogPrint(`Calculating ${m} ${occ} FINAL ${body_region_label} score...`);

                final_femur_score = Math.min(left_femur_comp_score, right_femur_comp_score);

                switch (occ) {
                    case "DRIVER":
                        left_femur_comp_score_weighted = left_femur_comp_score * driver_lower_leg_wt_factor;
                        right_femur_comp_score_weighted = right_femur_comp_score * driver_lower_leg_wt_factor;

                        final_femur_score = Math.min(left_femur_comp_score_weighted, right_femur_comp_score_weighted);

                        left_tibia_index_upper_score_weighted =
                            left_tibia_index_upper_score * driver_lower_leg_wt_factor;
                        right_tibia_index_upper_score_weighted =
                            right_tibia_index_upper_score * driver_lower_leg_wt_factor;
                        left_tibia_index_lower_score_weighted =
                            left_tibia_index_lower_score * driver_lower_leg_wt_factor;
                        right_tibia_index_lower_score_weighted =
                            right_tibia_index_lower_score * driver_lower_leg_wt_factor;

                        final_tibia_index_score = Math.min(
                            left_tibia_index_upper_score_weighted,
                            left_tibia_index_lower_score_weighted,
                            right_tibia_index_upper_score_weighted,
                            right_tibia_index_lower_score_weighted
                        );

                        left_tibia_comp_upper_score_weighted = left_tibia_comp_upper_score * driver_lower_leg_wt_factor;
                        right_tibia_comp_upper_score_weighted =
                            right_tibia_comp_upper_score * driver_lower_leg_wt_factor;
                        left_tibia_comp_lower_score_weighted = left_tibia_comp_lower_score * driver_lower_leg_wt_factor;
                        right_tibia_comp_lower_score_weighted =
                            right_tibia_comp_lower_score * driver_lower_leg_wt_factor;

                        final_tibia_comp_score = Math.min(
                            left_tibia_comp_upper_score_weighted,
                            left_tibia_comp_lower_score_weighted,
                            right_tibia_comp_upper_score_weighted,
                            right_tibia_comp_lower_score_weighted
                        );

                        brake_pedal_vertical_score_weighted = brake_pedal_vertical_score * driver_lower_leg_wt_factor;
                        brake_pedal_fore_aft_score_weighted = brake_pedal_fore_aft_score * driver_lower_leg_wt_factor;

                        lower_leg_final_score =
                            final_femur_score +
                            final_tibia_index_score +
                            final_tibia_comp_score +
                            brake_pedal_fore_aft_score_weighted +
                            brake_pedal_vertical_score_weighted;

                        break;

                    case "REAR_PASSENGER_SIDE":
                        left_femur_comp_score_weighted = left_femur_comp_score * passenger_lower_leg_wt_factor;
                        right_femur_comp_score_weighted = right_femur_comp_score * passenger_lower_leg_wt_factor;

                        final_femur_score = Math.min(left_femur_comp_score_weighted, right_femur_comp_score_weighted);

                        lower_leg_final_score = final_femur_score;

                        break;

                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Write variables for REPORTER */
            new Variable(
                templ,
                `${m}_${occ}_LEFT_FEMUR_COMPRESSION_SCORE_WEIGHTED`,
                "Weighted left femur compression score",
                left_femur_comp_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_RIGHT_FEMUR_COMPRESSION_SCORE_WEIGHTED`,
                "Weighted right femur compression score",
                right_femur_comp_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_FEMUR_FINAL_SCORE`,
                "Final femur score",
                final_femur_score.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_LOWER_LEG_FINAL_SCORE`,
                "Final lower leg score",
                lower_leg_final_score.toString(),
                "String",
                false,
                true
            );

            if (occ == "DRIVER") {
                new Variable(
                    templ,
                    `${m}_${occ}_LEFT_TIBIA_INDEX_UPPER_SCORE_WEIGHTED`,
                    "Weighted left tibia index upper score",
                    left_tibia_index_upper_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_RIGHT_TIBIA_INDEX_UPPER_SCORE_WEIGHTED`,
                    "Weighted right tibia index upper score",
                    right_tibia_index_upper_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_LEFT_TIBIA_INDEX_LOWER_SCORE_WEIGHTED`,
                    "Weighted left tibia index lower score",
                    left_tibia_index_lower_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_RIGHT_TIBIA_INDEX_LOWER_SCORE_WEIGHTED`,
                    "Weighted right tibia index lower score",
                    right_tibia_index_lower_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_TIBIA_INDEX_FINAL_SCORE`,
                    "Final tibia index score",
                    final_tibia_index_score.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_LEFT_TIBIA_COMPRESSION_UPPER_SCORE_WEIGHTED`,
                    "Weighted left tibia compression upper score",
                    left_tibia_comp_upper_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_RIGHT_TIBIA_COMPRESSION_UPPER_SCORE_WEIGHTED`,
                    "Weighted right tibia compression upper score",
                    right_tibia_comp_upper_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_LEFT_TIBIA_COMPRESSION_LOWER_SCORE_WEIGHTED`,
                    "Weighted left tibia compression lower score",
                    left_tibia_comp_lower_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_RIGHT_TIBIA_COMPRESSION_LOWER_SCORE_WEIGHTED`,
                    "Weighted right tibia compression lower score",
                    right_tibia_comp_lower_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_TIBIA_COMPRESSION_FINAL_SCORE`,
                    "Final tibia compression score",
                    final_tibia_comp_score.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_STRUCTURE_BRAKE_PEDAL_VERTICAL_INTRUSION_SCORE_WEIGHTED`,
                    "Weighted brake pedal vertical intrusion score",
                    brake_pedal_vertical_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_STRUCTURE_BRAKE_PEDAL_FORE_AFT_INTRUSION_SCORE_WEIGHTED`,
                    "Weighted brake pedal fore-aft intrusion score",
                    brake_pedal_fore_aft_score_weighted.toString(),
                    "String",
                    false,
                    true
                );
            }
        }
    }
}
