/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest score calculation for the specified models
 */
function chest_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "REAR_PASSENGER_SIDE"];

    /* Weighting factors */
    let driver_chest_wt_factor = 0.923;
    let passenger_chest_wt_factor = 0.8;

    new Variable(
        templ,
        "DRIVER_CHEST_WEIGHTING_FACTOR",
        "Driver chest weighting factor",
        driver_chest_wt_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "REAR_PASSENGER_SIDE_CHEST_WEIGHTING_FACTOR",
        "Rear passenger side chest weighting factor",
        passenger_chest_wt_factor.toString(),
        "String",
        false,
        true
    );

    for (let m of models) {
        for (let occ of occupants) {
            let body_region_label = `Chest`;
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Chest score and modifiers */

            /* Driver */
            let steering_column_backward_displacement_score;
            let chest_injury_criterion_score;
            let chest_secondary_contact_modifier;

            //Initialising chest_injury_criterion_modifier to 0

            /* Shared */
            let chest_comp_score;

            if (occ == "DRIVER") {
                chest_comp_score = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_SCORE`, "float");
                steering_column_backward_displacement_score = get_variable_value(
                    status,
                    `${m}_STRUCTURE_STEERING_COLUMN_FORE_AFT_INTRUSION_SCORE`,
                    "float"
                );
                chest_injury_criterion_score = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_THREE_MS_EXCEEDENCE_SCORE`,
                    "float"
                );
                chest_secondary_contact_modifier = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_SECONDARY_STEERING_WHEEL_CONTACT_MODIFIER`,
                    "float",
                    true
                );
            } else if (occ == "REAR_PASSENGER_SIDE") {
                chest_comp_score = get_variable_value(
                    status,
                    `${m}_${occ}_CHEST_COMPRESSION_REAR_PASSENGER_SCORE`,
                    "float"
                );
            } else {
                LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let chest_comp_score_weighted = 0;
            let steering_column_backward_displacement_score_weighted = 0;
            let chest_injury_criterion_score_weighted = 0;
            let chest_secondary_contact_modifier_weighted = 0;

            let chest_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                switch (occ) {
                    case "DRIVER":
                        chest_comp_score_weighted = chest_comp_score * driver_chest_wt_factor;
                        steering_column_backward_displacement_score_weighted =
                            steering_column_backward_displacement_score * driver_chest_wt_factor;
                        chest_injury_criterion_score_weighted = chest_injury_criterion_score * driver_chest_wt_factor;
                        chest_secondary_contact_modifier_weighted =
                            chest_secondary_contact_modifier * driver_chest_wt_factor;

                        chest_score =
                            chest_comp_score_weighted +
                            steering_column_backward_displacement_score_weighted +
                            chest_injury_criterion_score_weighted +
                            chest_secondary_contact_modifier_weighted;

                        chest_score = Math.max(chest_score, 0);

                        LogPrint(`Final ${m} ${occ} ${body_region_label} score = ${chest_score}`);

                        break;

                    case "REAR_PASSENGER_SIDE":
                        chest_comp_score_weighted = chest_comp_score * passenger_chest_wt_factor;

                        chest_score = chest_comp_score_weighted;
                        LogPrint(`${m} ${occ} ${body_region_label} score = ${chest_score}`);

                        break;

                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Weighted scores */
            new Variable(
                templ,
                `${m}_${occ}_CHEST_COMPRESSION_SCORE_WEIGHTED`,
                "Weighted chest compression score",
                chest_comp_score_weighted.toString(),
                "String",
                false,
                true
            );

            if (occ == "DRIVER") {
                new Variable(
                    templ,
                    `${m}_STRUCTURE_STEERING_COLUMN_FORE_AFT_INTRUSION_SCORE_WEIGHTED`,
                    "Weighted steering column backward displacement score",
                    steering_column_backward_displacement_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_CHEST_THREE_MS_EXCEEDENCE_SCORE_WEIGHTED`,
                    "Weighted chest injury criterion score",
                    chest_injury_criterion_score_weighted.toString(),
                    "String",
                    false,
                    true
                );

                new Variable(
                    templ,
                    `${m}_${occ}_CHEST_SECONDARY_STEERING_WHEEL_CONTACT_MODIFIER_WEIGHTED`,
                    "Weighted chest secondary contact modifier",
                    chest_secondary_contact_modifier_weighted.toString(),
                    "String",
                    false,
                    true
                );
            }

            /*Sending values to reporter*/
            new Variable(
                templ,
                `${m}_${occ}_CHEST_FINAL_SCORE`,
                "Final chest score",
                chest_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
