/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
abdomen_score();

/**
 * Performs the abdomen score calculation for the specified models
 */
function abdomen_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    /* Rear Passenger only */
    let occupants = ["REAR_PASSENGER_SIDE"];

    /* Weighting factors */
    let passenger_abdomen_wt_factor = 0.8;

    new Variable(
        templ,
        "REAR_PASSENGER_SIDE_ABDOMEN_WEIGHTING_FACTOR",
        "Abdomen weighting factor",
        passenger_abdomen_wt_factor.toString(),
        "String",
        false,
        true
    );

    for (let m of models) {
        for (let occ of occupants) {
            let body_region_label = `Abdomen`;
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Rear Passenger */
            let left_ilium_load_rate_modifier = get_variable_value(
                status,
                `${m}_${occ}_LEFT_ILIUM_LOAD_RATE_MODIFIER`,
                "float",
                true
            );

            let right_ilium_load_rate_modifier = get_variable_value(
                status,
                `${m}_${occ}_RIGHT_ILIUM_LOAD_RATE_MODIFIER`,
                "float",
                true
            );

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let left_ilium_load_rate_modifier_weighted = 0;
            let right_ilium_load_rate_modifier_weighted = 0;

            let base_abdomen_score = 4; /* Initial score before subtracting modifiers */
            let base_abdomen_score_weighted = base_abdomen_score * passenger_abdomen_wt_factor;

            let abdomen_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                switch (occ) {
                    case "REAR_PASSENGER_SIDE":
                        left_ilium_load_rate_modifier_weighted =
                            left_ilium_load_rate_modifier * passenger_abdomen_wt_factor;
                        right_ilium_load_rate_modifier_weighted =
                            right_ilium_load_rate_modifier * passenger_abdomen_wt_factor;

                        abdomen_score =
                            (base_abdomen_score + left_ilium_load_rate_modifier + right_ilium_load_rate_modifier) *
                            passenger_abdomen_wt_factor;

                        LogPrint(`${m} ${occ} ${body_region_label} score = ${abdomen_score}`);
                        break;

                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Base score */
            new Variable(
                templ,
                `${m}_${occ}_BASE_ABDOMEN_SCORE`,
                "Abdomen Base Score",
                base_abdomen_score.toString(),
                "String",
                false,
                true
            );
            new Variable(
                templ,
                `${m}_${occ}_BASE_ABDOMEN_WEIGHTED_SCORE`,
                "Abdomen Base Weighted Score",
                base_abdomen_score_weighted.toString(),
                "String",
                false,
                true
            );

            /* Weighted scores */
            new Variable(
                templ,
                `${m}_${occ}_LEFT_ILIUM_LOAD_RATE_MODIFIER_WEIGHTED`,
                "Weighted left ilium load rate modifier",
                left_ilium_load_rate_modifier_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_RIGHT_ILIUM_LOAD_RATE_MODIFIER_WEIGHTED`,
                "Weighted right ilium load rate modifier",
                right_ilium_load_rate_modifier_weighted.toString(),
                "String",
                false,
                true
            );

            /*Sending values to reporter*/
            new Variable(
                templ,
                `${m}_${occ}_ABDOMEN_FINAL_SCORE`,
                "Final abdomen score",
                abdomen_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
