/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
neck_score();

/**
 * Performs the neck score calculation for the specified models
 */
function neck_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `neck`;

    /* Weighting factors */
    let driver_neck_weighting_factor = 0.231;
    let passenger_neck_weighting_factor = 0.2;

    new Variable(
        templ,
        "DRIVER_NECK_WEIGHTING_FACTOR",
        "Driver head weighting factor",
        driver_neck_weighting_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_NECK_WEIGHTING_FACTOR",
        "Front passenger head weighting factor",
        passenger_neck_weighting_factor.toString(),
        "String",
        false,
        true
    );

    let driver_neck_max_limit = 0.924;
    let passenger_neck_max_limit = 0.8;

    new Variable(
        templ,
        "DRIVER_NECK_MAX_POINTS",
        "Driver neck max points",
        driver_neck_max_limit.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_NECK_MAX_POINTS",
        "Front passenger neck max points",
        passenger_neck_max_limit.toString(),
        "String",
        false,
        true
    );

    let occupants = ["DRIVER", "FRONT_PASSENGER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Neck score and modifiers */

            /*Shared*/
            let neck_axial_tension_score;
            let neck_shear_score;
            let neck_extension_score;

            switch (occ) {
                case "DRIVER":
                    neck_axial_tension_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_TENSION_EXCEEDENCE_SCORE`,
                        "float"
                    );
                    neck_shear_score = get_variable_value(status, `${m}_${occ}_NECK_SHEAR_EXCEEDENCE_SCORE`, "float");
                    neck_extension_score = get_variable_value(status, `${m}_${occ}_NECK_EXTENSION_SCORE`, "float");
                    break;

                case "FRONT_PASSENGER":
                    neck_axial_tension_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_AXIAL_PASSENGER_SCORE`,
                        "float"
                    );
                    neck_shear_score = get_variable_value(status, `${m}_${occ}_NECK_SHEAR_PASSENGER_SCORE`, "float");
                    neck_extension_score = get_variable_value(
                        status,
                        `${m}_${occ}_NECK_EXTENSION_PASSENGER_SCORE`,
                        "float"
                    );
                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let neck_tension_score_weighted = 0;
            let neck_shear_score_weighted = 0;
            let neck_extension_score_weighted = 0;

            let neck_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                switch (occ) {
                    case "DRIVER":
                        neck_tension_score_weighted = neck_axial_tension_score * driver_neck_weighting_factor;
                        neck_shear_score_weighted = neck_shear_score * driver_neck_weighting_factor;
                        neck_extension_score_weighted = neck_extension_score * driver_neck_weighting_factor;
                        neck_score = Math.min(
                            neck_tension_score_weighted,
                            neck_shear_score_weighted,
                            neck_extension_score_weighted
                        );

                        /* Bound the score between upper and lower limits */
                        neck_score = Math.max(Math.min(neck_score, driver_neck_max_limit), 0.0);

                        break;

                    case "FRONT_PASSENGER":
                        neck_tension_score_weighted = neck_axial_tension_score * passenger_neck_weighting_factor;
                        neck_shear_score_weighted = neck_shear_score * passenger_neck_weighting_factor;
                        neck_extension_score_weighted = neck_extension_score * passenger_neck_weighting_factor;
                        neck_score = Math.min(
                            neck_tension_score_weighted,
                            neck_shear_score_weighted,
                            neck_extension_score_weighted
                        );
                        /* Bound the score between upper and lower limits */
                        neck_score = Math.max(Math.min(neck_score, passenger_neck_max_limit), 0.0);

                        break;

                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }

                LogPrint(`${m} ${occ} neck score = ${neck_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Weighted scores */
            new Variable(
                templ,
                `${m}_${occ}_NECK_TENSION_SCORE_WEIGHTED`,
                "Neck tension score weighted",
                neck_tension_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_NECK_SHEAR_SCORE_WEIGHTED`,
                "Neck shear score weighted",
                neck_shear_score_weighted.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_NECK_EXTENSION_SCORE_WEIGHTED`,
                "Neck extension score weighted",
                neck_extension_score_weighted.toString(),
                "String",
                false,
                true
            );

            /* Overall scores */

            new Variable(
                templ,
                `${m}_${occ}_NECK_FINAL_SCORE`,
                "Final neck score",
                neck_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
