/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_score();

/**
 * Performs the head score calculation for the specified models
 */
function head_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `head`;

    /* Weighting factors */
    let driver_head_weighting_factor = 0.923;
    let passenger_head_weighting_factor = 0.8;

    new Variable(
        templ,
        "DRIVER_HEAD_WEIGHTING_FACTOR",
        "Driver head weighting factor",
        driver_head_weighting_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_HEAD_WEIGHTING_FACTOR",
        "Front passenger head weighting factor",
        passenger_head_weighting_factor.toString(),
        "String",
        false,
        true
    );

    let driver_head_max_limit = 3.692;
    let passenger_head_max_limit = 3.2;

    new Variable(
        templ,
        "DRIVER_HEAD_MAX_POINTS",
        "Driver head max points",
        driver_head_max_limit.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_HEAD_MAX_POINTS",
        "Front passenger head max points",
        passenger_head_max_limit.toString(),
        "String",
        false,
        true
    );

    let occupants = ["DRIVER", "FRONT_PASSENGER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Head score and modifiers */

            /*Driver */
            let head_hic36_score;
            let steering_column_intrusion_z_score;

            /*Front passenger */
            let head_hic15_score;

            switch (occ) {
                case "DRIVER":
                    head_hic36_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC36_SCORE`, "float");
                    steering_column_intrusion_z_score = get_variable_value(
                        status,
                        `${m}_STRUCTURE_STEERING_COLUMN_VERTICAL_INTRUSION_SCORE`,
                        "float"
                    );
                    break;

                case "FRONT_PASSENGER":
                    head_hic15_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC15_SCORE`, "float");
                    break;

                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let head_score = 0;

            let head_hic_score_weighted = 0;
            let steering_column_intrusion_z_score_weighted = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                switch (occ) {
                    case "DRIVER":
                        head_hic_score_weighted = head_hic36_score * driver_head_weighting_factor;
                        steering_column_intrusion_z_score_weighted =
                            steering_column_intrusion_z_score * driver_head_weighting_factor;

                        head_score = head_hic_score_weighted + steering_column_intrusion_z_score_weighted;
                        /* Bound the score between upper and lower limits */
                        head_score = Math.max(Math.min(head_score, driver_head_max_limit), 0.0);

                        break;
                    case "FRONT_PASSENGER":
                        head_hic_score_weighted = head_hic15_score * passenger_head_weighting_factor;
                        head_score = head_hic_score_weighted;

                        /* Bound the score between upper and lower limits */
                        head_score = Math.max(Math.min(head_score, passenger_head_max_limit), 0.0);
                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }

                LogPrint(`${m} ${occ} head score = ${head_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /*Weighted Score*/
            new Variable(
                templ,
                `${m}_${occ}_HEAD_HIC_SCORE_WEIGHTED`,
                "Head HIC score weighted",
                head_hic_score_weighted.toString(),
                "String",
                false,
                true
            );

            if (occ == "DRIVER") {
                new Variable(
                    templ,
                    `${m}_${occ}_STEERING_COLUMN_UPPER_DISPLACEMENT_SCORE_WEIGHTED`,
                    "Steering column upper displacement score weighted",
                    steering_column_intrusion_z_score_weighted.toString(),
                    "String",
                    false,
                    true
                );
            }

            /* Overall scores */
            new Variable(
                templ,
                `${m}_${occ}_HEAD_FINAL_SCORE`,
                "Final head score",
                head_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
