/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_and_abdomen_score();

/**
 * Performs the chest  score calculation for the specified models
 */
function chest_and_abdomen_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `chest and abdomen`;

    let driver_chest_weighting_factor = 0.923;
    let passenger_chest_abdomen_weighting_factor = 0.8;

    new Variable(
        templ,
        "DRIVER_CHEST_WEIGHTING_FACTOR",
        "Driver chest weighting factor",
        driver_chest_weighting_factor.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_CHEST_ABDOMEN_WEIGHTING_FACTOR",
        "Front passenger chest weighting factor",
        passenger_chest_abdomen_weighting_factor.toString(),
        "String",
        false,
        true
    );

    let driver_chest_max_limit = 3.692;
    let passenger_chest_max_limit = 3.2;
    let passenger_abdomen_max_limit = 3.2;
    let passenger_chest_abdomen_max_limit = passenger_chest_max_limit + passenger_abdomen_max_limit;

    new Variable(
        templ,
        "DRIVER_CHEST_MAX_POINTS",
        "Driver chest max points",
        driver_chest_max_limit.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_CHEST_MAX_POINTS",
        "Front passenger chest max points",
        passenger_chest_max_limit.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_ABDOMEN_MAX_POINTS",
        "Front passenger abdomen max points",
        passenger_abdomen_max_limit.toString(),
        "String",
        false,
        true
    );

    new Variable(
        templ,
        "FRONT_PASSENGER_CHEST_ABDOMEN_MAX_POINTS",
        "Front passenger chest abdomen max points",
        passenger_chest_abdomen_max_limit.toString(),
        "String",
        false,
        true
    );

    let occupants = ["DRIVER", "FRONT_PASSENGER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Chest score and modifiers */

            /* Shared */
            let chest_comp_score;

            /* Driver */
            let steering_wheel_contact_mod;
            let chest_injury_criterion_score;
            let steering_column_fore_aft_intrusion;

            /* Front Passenger */
            let left_iliac_modifier;
            let right_iliac_modifier;

            switch (occ) {
                case "DRIVER":
                    chest_comp_score = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_SCORE`, "float");

                    steering_column_fore_aft_intrusion = get_variable_value(
                        status,
                        `${m}_STRUCTURE_STEERING_COLUMN_FORE_AFT_INTRUSION_SCORE`,
                        "float"
                    );

                    steering_wheel_contact_mod = get_variable_value(
                        status,
                        `${m}_${occ}_CHEST_STEERING_WHEEL_CONTACT_MODIFIER`,
                        "int"
                    );
                    chest_injury_criterion_score = get_variable_value(
                        status,
                        `${m}_${occ}_CHEST_THREE_MS_EXCEEDENCE_SCORE`,
                        "float"
                    );

                    break;
                case "FRONT_PASSENGER":
                    chest_comp_score = get_variable_value(
                        status,
                        `${m}_${occ}_CHEST_COMPRESSION_PASSENGER_SCORE`,
                        "float"
                    );

                    left_iliac_modifier = get_variable_value(status, `${m}_${occ}_LEFT_ILIAC_LOAD_MODIFIER`, "float");

                    right_iliac_modifier = get_variable_value(status, `${m}_${occ}_RIGHT_ILIAC_LOAD_MODIFIER`, "float");

                    break;
                default:
                    LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                    Exit();
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let chest_score = 0;
            let abdomen_score = 0;
            let passenger_abdomen = 4;
            let passenger_abdomen_weighting = 0;
            let chest_abdomen_score = 0;
            let chest_comp_wt_score = 0;
            let steering_column_wt_score = 0;
            let left_iliac_wt_score = 0;
            let right_iliac_wt_score = 0;

            /* If we have all the required variables, calculate the final scores */
            if (status.success) {
                /* Score calculation is different for different occupants */
                switch (occ) {
                    case "DRIVER":
                        chest_comp_wt_score = chest_comp_score * driver_chest_weighting_factor;
                        steering_column_wt_score = steering_column_fore_aft_intrusion * driver_chest_weighting_factor;
                        chest_score = chest_comp_wt_score + steering_column_wt_score;

                        /* Bound the score between upper and lower limits */
                        chest_score = Math.max(
                            Math.min(
                                chest_score + chest_injury_criterion_score + steering_wheel_contact_mod,
                                driver_chest_max_limit
                            ),
                            0.0
                        );

                        chest_abdomen_score = chest_score;

                        break;

                    case "FRONT_PASSENGER":
                        /* Chest Score */
                        chest_comp_wt_score = chest_comp_score * passenger_chest_abdomen_weighting_factor;
                        chest_score = chest_comp_wt_score;

                        /* Abdomen Score */
                        left_iliac_wt_score = left_iliac_modifier * passenger_chest_abdomen_weighting_factor;
                        right_iliac_wt_score = right_iliac_modifier * passenger_chest_abdomen_weighting_factor;
                        passenger_abdomen_weighting = passenger_abdomen * passenger_chest_abdomen_weighting_factor;
                        abdomen_score = passenger_abdomen_weighting + left_iliac_wt_score + right_iliac_wt_score;

                        /* Bound the score between upper and lower limits */
                        chest_score = Math.max(Math.min(chest_score, passenger_chest_max_limit), 0.0);
                        abdomen_score = Math.max(Math.min(abdomen_score, passenger_abdomen_max_limit), 0.0);

                        chest_abdomen_score = chest_score + abdomen_score;

                        break;

                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }

                LogPrint(`${m} ${occ} chest score = ${chest_score}`);
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Shared */
            new Variable(
                templ,
                `${m}_${occ}_CHEST_COMPRESSION_WEIGHTED_SCORE`,
                "Chest compression weighting",
                chest_comp_wt_score.toString(),
                "String",
                false,
                true
            );

            /* Driver */
            if (occ == "DRIVER") {
                new Variable(
                    templ,
                    `${m}_STEERING_COLUMN_WEIGHTED_SCORE`,
                    "Steering column weighting",
                    steering_column_wt_score.toString(),
                    "String",
                    false,
                    true
                );
            }

            /* Front Passenger */
            if (occ == "FRONT_PASSENGER") {
                new Variable(
                    templ,
                    `${m}_${occ}_BASE_ABDOMEN_SCORE`,
                    "Abdomen Base Score",
                    passenger_abdomen.toString(),
                    "String",
                    false,
                    true
                );
                new Variable(
                    templ,
                    `${m}_${occ}_BASE_ABDOMEN_WEIGHTED_SCORE`,
                    "Abdomen Base Weighted Score",
                    passenger_abdomen_weighting.toString(),
                    "String",
                    false,
                    true
                );
                new Variable(
                    templ,
                    `${m}_${occ}_LEFT_ILIAC_WEIGHTED_SCORE`,
                    "Left Iliac Weighted Score",
                    left_iliac_wt_score.toString(),
                    "String",
                    false,
                    true
                );
                new Variable(
                    templ,
                    `${m}_${occ}_RIGHT_ILIAC_WEIGHTED_SCORE`,
                    "Right Iliac Weighted Score",
                    right_iliac_wt_score.toString(),
                    "String",
                    false,
                    true
                );
                new Variable(
                    templ,
                    `${m}_${occ}_ABDOMEN_FINAL_SCORE`,
                    "Front Passenger Abdomen Score",
                    abdomen_score.toString(),
                    "String",
                    false,
                    true
                );
            }

            /* Overall scores */

            new Variable(
                templ,
                `${m}_${occ}_CHEST_FINAL_SCORE`,
                "Final chest score",
                chest_score.toString(),
                "String",
                false,
                true
            );

            new Variable(
                templ,
                `${m}_${occ}_CHEST_ABDOMEN_FINAL_SCORE`,
                "Final chest abdomen score",
                chest_abdomen_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
