/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
rating_dial_1();

/**
 * Draw rating dial images for the specified models
 */
function rating_dial_1() {
    var templ = Template.GetCurrent();
    let m = get_model_list();
    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    let occ;
    /* Create a status object to track whether REPORTER Variables are all present and valid.
     * <success> is initially true but will be set to false if anything missing or invalid. */
    let status = { success: true, missing: [], invalid: [] };
    if (!output_dir) {
        LogError(`Could not find OUTPUT_DIR variable for writing occupant image files.`);
        return;
    } else if (!File.IsDirectory(output_dir)) {
        LogError(`OUTPUT_DIR is not a valid directory: ${output_dir}`);
        return;
    }

    var GREEN_COL = get_variable_value(status, "COLOUR_IIHS_GREEN"); // Green
    var YELLOW_COL = get_variable_value(status, "COLOUR_IIHS_YELLOW"); // Yellow
    var ORANGE_COL = get_variable_value(status, "COLOUR_IIHS_ORANGE"); // Orange
    var RED_COL = get_variable_value(status, "COLOUR_IIHS_RED"); // Red

    var start = get_variable_value(status, "BAND_0_IIHS"); // max
    var good = get_variable_value(status, "BAND_1_IIHS"); // max
    var acceptable = get_variable_value(status, "BAND_2_IIHS"); // max
    var marginal = get_variable_value(status, "BAND_3_IIHS"); // max
    var poor = get_variable_value(status, "BAND_4_IIHS"); // max

    //IIHS Colours
    // var GREEN_COL  = '#52B442';   // Green
    // var YELLOW_COL = '#FCE702';   // Yellow
    // var ORANGE_COL = '#F79800';   // Orange
    // var RED_COL    = '#F40000';   // Red

    //overall score
    var score = parseInt(templ.GetVariableValue("M1_TOTAL_WEIGHT"));

    //constants
    var PIXELS_PER_REPORTER_UNIT = 72;

    //image size #note this can be overwritten if image file found
    var width_units = 60;
    var height_units = 30;

    //    //regenerate image object so that dial moves
    var regenerate = true;
    var i_draw_rating_dial = Item.GetFromName(templ.GetPage(0), "Rating Dial");

    if (i_draw_rating_dial == null) {
        LogWarning(`Could not find image file item named Rating Dial on page 1\n
                The image will still be created but it will have a default size\n
                and will not be Regenerated if it is referenced by another image file item`);

        regenerate = false;
    } else {
        //take size based on actual REPORTER item size, but make sure it is always big enough to draw everything so take max
        width_units = Math.max(width_units, i_draw_rating_dial.width);
        height_units = Math.max(height_units, i_draw_rating_dial.height);
    }

    var width = width_units * PIXELS_PER_REPORTER_UNIT;
    var height = height_units * PIXELS_PER_REPORTER_UNIT;

    //find constraining dimension

    var tick_length = 1.25;
    var tick_radius_border = 1.25;
    var text_radius_border = 0.75;
    var height_offset = 4;
    var border = 3;
    var border_units = border + tick_length + tick_radius_border + text_radius_border; // 1mm general border, 5mm for text, 1mm

    var usable_width = width_units - 2 * border_units;
    var usable_height = height_units - border_units - height_offset;
    //todo check usable radius is >0
    if (usable_width > 2 * usable_height) {
        //then height is constraining usable radius is height
        var usable_radius = usable_height;
    } else {
        var usable_radius = usable_width / 2;
    }

    // dial sizing
    var radius_outer = usable_radius * PIXELS_PER_REPORTER_UNIT;
    var radius_inner = 0.5 * radius_outer;

    // text parameters
    var target_font_size = Math.round(4 * (25.4 / 72) * PIXELS_PER_REPORTER_UNIT); //PIXELS 25.4mm in an inch and font points are ~1/72 inches and reporter unit is 1mm
    LogPrint("target_font_size =" + target_font_size + "px");
    var text_radius = radius_outer + (border_units - border) * PIXELS_PER_REPORTER_UNIT;

    //tick line parameters
    var tick_line = {};
    tick_line.width = PIXELS_PER_REPORTER_UNIT / 2;
    tick_line.radius = radius_outer + tick_radius_border * PIXELS_PER_REPORTER_UNIT; //width*0.37;
    tick_line.tick_length = tick_length * PIXELS_PER_REPORTER_UNIT;
    tick_line.tick_width = PIXELS_PER_REPORTER_UNIT / 4;
    tick_line.colour = "#647382"; //'black';//'#F3F7FB'; //"#647382";//
    tick_line.lineStyle = Reporter.LINE_SOLID;
    tick_line.tick_points = [start, good, acceptable, marginal, poor];
    tick_line.end = poor;
    tick_line.step_size = 0.001;

    // Draw images of coloured in driver and save to file
    draw_rating_dial(score, start, good, acceptable, marginal, poor);

    //regenerate the "Rating Dial" image item if it exists
    if (regenerate) i_draw_rating_dial.Generate();

    /**
     * Calculates the x-coordinate for a point on the circumference of a circle
     * @param {number} fraction The fraction of the circle (0 to 1)
     * @param {number} radius The radius of the circle
     * @param {number} x_offset The x-coordinate offset from the center
     * @returns {number} The x-coordinate value
     */
    function getX(fraction, radius, x_offset) {
        return Math.round(Math.cos(-Math.PI * fraction) * radius + x_offset);
    }

    /**
     * Calculates the y-coordinate for a point on the circumference of a circle
     * @param {number} fraction The fraction of the circle (0 to 1)
     * @param {number} radius The radius of the circle
     * @param {number} y_offset The y-coordinate offset from the center
     * @returns {number} The y-coordinate value
     */
    function getY(fraction, radius, y_offset) {
        return Math.round(Math.sin(-Math.PI * fraction) * radius + y_offset);
    }

    /**
     * Generates an array of coordinates for points forming a polygon strip along the circumference of a dial
     * @param {number} fraction_start The starting fraction of the dial (0 to 1)
     * @param {number} fraction_end The ending fraction of the dial (0 to 1)
     * @param {number} x_offset The x-coordinate offset from the center
     * @param {number} y_offset The y-coordinate offset from the center
     * @returns {number[]} An array of x and y coordinates representing points on the dial's circumference
     */
    function get_dial_coords(fraction_start, fraction_end, x_offset, y_offset) {
        //this function returns and array of all the points making the polygon for
        //a strip of the dialmeter

        var step_size = 0.001; //increment in 0.1% steps to make a smooth circle
        var poly_array = [];

        //go anticlockwise on inner side
        poly_array = poly_array.concat(
            getCircumCoords(step_size, fraction_start, fraction_end, radius_inner, x_offset, y_offset)
        );

        //jump to outer side in clockwise direction (i.e. from end to start with radius_outer)
        poly_array = poly_array.concat(
            getCircumCoords(step_size, fraction_end, fraction_start, radius_outer, x_offset, y_offset)
        );

        //polygon auto closes so no need to add initial point.
        //LogPrint("fraction_start: " + fraction_start + " fraction_end: " + fraction_end );
        //LogPrint(JSON.stringify(poly_array));
        return poly_array;
    }

    /**
     * Generates an array of coordinates for points along the circumference of a circle (the rating dial)
     * @param {number} step_size The angular step size (in radians) for generating points
     * @param {number} fraction_start The starting fraction of the circle (0 to 1)
     * @param {number} fraction_end The starting fraction of the circle (0 to 1)
     * @param {number} radius The radius of the circle
     * @param {number} x_offset The x-coordinate offset from the center.
     * @param {number} y_offset The y-coordinate offset from the center.
     * @returns {number[]} An array of x and y coordinates representing points on the circumference.
     */
    function getCircumCoords(step_size, fraction_start, fraction_end, radius, x_offset, y_offset) {
        //reduce step_size to make a smoother circle
        //note that:
        //fraction_end>fraction_start gives anticlockwise points and
        //fraction_end<fraction_start gives clockwise points.

        var sign = 1;
        if (fraction_end < fraction_start) sign = -1;

        var fraction = fraction_start;
        step_size *= sign;

        var poly_array = [];

        while (fraction * sign < fraction_end * sign) {
            poly_array.push(getX(fraction, radius, x_offset), getY(fraction, radius, y_offset));
            fraction += step_size;
        }
        fraction = fraction_end;
        poly_array.push(getX(fraction, radius, x_offset), getY(fraction, radius, y_offset));
        return poly_array;
    }

    /**
     * Draws a rating dial with different colored bands based on specified thresholds
     * @param {number} score The overall weight (within the range of start to poor)
     * @param {number} start The starting point of the dial (minimum possible score)
     * @param {number} good The threshold for the "good" region
     * @param {number} acceptable The threshold for the "acceptable" region
     * @param {number} marginal The threshold for the "marginal" region
     * @param {number} poor The ending point of the dial (maximum possible score)
     */
    function draw_rating_dial(score, start, good, acceptable, marginal, poor) {
        // Get the default directory where we will write the images

        var template = Template.GetCurrent();
        var dir = template.GetVariableValue("IMAGES_DIR");

        var img = new Image(width, height);

        img.lineWidth = 0;

        // Draw the dial

        var end = poor;

        var x_offset = width / 2;
        var y_offset = height - height_offset * PIXELS_PER_REPORTER_UNIT;

        var begin_band, end_band;

        img.lineColour = GREEN_COL;
        img.fillColour = GREEN_COL;
        begin_band = start / end;
        end_band = good / end;
        img.Polygon(get_dial_coords(begin_band, end_band, x_offset, y_offset));
        writeTextOnDial(img, begin_band, text_radius, start, target_font_size, x_offset, y_offset);

        img.lineColour = YELLOW_COL;
        img.fillColour = YELLOW_COL;
        begin_band = end_band;
        end_band = acceptable / end;
        img.Polygon(get_dial_coords(begin_band, end_band, x_offset, y_offset));
        writeTextOnDial(img, begin_band, text_radius, good, target_font_size, x_offset, y_offset);

        img.lineColour = ORANGE_COL;
        img.fillColour = ORANGE_COL;
        begin_band = end_band;
        end_band = marginal / end;
        img.Polygon(get_dial_coords(begin_band, end_band, x_offset, y_offset));
        writeTextOnDial(img, begin_band, text_radius, acceptable, target_font_size, x_offset, y_offset);

        img.lineColour = RED_COL;
        img.fillColour = RED_COL;
        begin_band = end_band;
        end_band = poor / end;
        img.Polygon(get_dial_coords(begin_band, end_band, x_offset, y_offset));
        writeTextOnDial(img, begin_band, text_radius, marginal, target_font_size, x_offset, y_offset);

        writeTextOnDial(img, end_band, text_radius, poor, target_font_size, x_offset, y_offset);

        // Draw the dial line to indicate where the rating is on the bar

        // draw dial coords first on the horizontal pointing to the right, then rotate based on score
        var l = (radius_outer - radius_inner) * 0.8 + radius_inner; //length of dial so that it falls 80% to edge of the spedo
        var t = 2 * PIXELS_PER_REPORTER_UNIT;

        var dial_coords = [0, -t / 2, l * 0.4, -t / 2, l, -t * 0.1, l, t * 0.1, l * 0.4, t / 2, 0, t / 2];

        LogPrint("Draw Dial. Score = " + score);

        for (var i = 0; i < dial_coords.length; i += 2) {
            var r = Math.sqrt(dial_coords[i] * dial_coords[i] + dial_coords[i + 1] * dial_coords[i + 1]);
            var x = dial_coords[i];
            var y = dial_coords[i + 1];
            var theta = (-Math.PI * score) / end;
            dial_coords[i] = Math.round(x * Math.cos(theta) - y * Math.sin(theta) + x_offset);
            dial_coords[i + 1] = Math.round(x * Math.sin(theta) + y * Math.cos(theta) + y_offset);
        }

        //LogPrint(JSON.stringify(dial_coords));

        img.lineColour = "black";
        img.fillColour = "black";
        img.Polygon(dial_coords);
        var ellipse_radius = t / 2;
        img.Ellipse(
            x_offset - ellipse_radius,
            y_offset - ellipse_radius,
            x_offset + ellipse_radius,
            y_offset + ellipse_radius
        );

        //draw spedo tick line
        img.lineColour = tick_line.colour;
        img.lineWidth = tick_line.width;
        img.lineStyle = tick_line.lineStyle;
        img.Polyline(getCircumCoords(tick_line.step_size, 0, 1, tick_line.radius, x_offset, y_offset));
        drawTick(img, tick_line, x_offset, y_offset);

        // Write out image

        //img.Save(dir + "/rating_dial.png", Image.PNG);
        img.Save(`${output_dir}/${m}_rating_dial.png`, Image.PNG);
    }

    /**
     * Draws ticks on a dial at specified positions
     * @param {Image} img The image on which to draw the tick
     * @param {*} tick_line Configuration for the tick line (width, radius, tick length, tick width, colour, line style, etc)
     * @param {number} x_offset The x-coordinate offset from the center
     * @param {number} y_offset The y-coordinate offset from the center
     */
    function drawTick(img, tick_line, x_offset, y_offset) {
        img.lineColour = tick_line.colour;
        img.lineWidth = tick_line.tick_width;
        img.lineStyle = tick_line.lineStyle;

        var start_radius = tick_line.radius;

        for (var i = 0; i < tick_line.tick_points.length; i++) {
            var poly_line_array = [];

            var fraction = tick_line.tick_points[i] / tick_line.end;
            LogPrint("Draw tick at " + tick_line.tick_points[i]);

            poly_line_array.push(getX(fraction, start_radius, x_offset), getY(fraction, start_radius, y_offset));
            poly_line_array.push(
                getX(fraction, start_radius + tick_line.tick_length, x_offset),
                getY(fraction, start_radius + tick_line.tick_length, y_offset)
            );

            img.Polyline(poly_line_array);
        }
    }

    /**
     * Writes text on a dial at a specified position
     * @param {Image} img The image on which to write the text
     * @param {number} fraction The position on the dial (0 to 1)
     * @param {number} text_radius The radius at which to place the text
     * @param {string|number} text The text to be written (converted to a string)
     * @param {number} text_height The height of the text
     * @param {number} x_offset The x-coordinate offset from the center
     * @param {number} y_offset The y-coordinate offset from the center
     */
    function writeTextOnDial(img, fraction, text_radius, text, text_height, x_offset, y_offset) {
        //make sure text is a string
        text = text.toString();
        //var l = text.length;

        try {
            img.fontSize = pixels_to_font_size(target_font_size);
        } catch (err) {
            LogPrint("Failed to call pixels_to_font_size()");
            LogPrint(err);
            img.fontSize = target_font_size;
        }

        img.fontStyle = Reporter.TEXT_NORMAL;
        img.font = "Segoe UI";
        img.fontColour = "black"; //#f3f7fb";

        img.fontJustify = Reporter.JUSTIFY_LEFT;

        var scaling;

        if (fraction > 0.5) {
            scaling = 1 - fraction;
            img.fontJustify = Reporter.JUSTIFY_RIGHT;
        } else scaling = fraction;

        if (scaling > 0.45) {
            scaling = 1 - fraction;
            img.fontJustify = Reporter.JUSTIFY_CENTRE;
        }

        //assume text width is 0.8 times height
        var text_x_offset = 0; //l*text_height*0.8/2;
        var text_y_offset = (0.5 - scaling) * text_height;

        //add dropshadow
        // var shadow_border = -1;
        // if (shadow_border>=0)
        // {
        //     var currentFontSize = img.fontSize;
        //     var scale = (currentFontSize+shadow_border)/currentFontSize;
        //     img.fontSize+=shadow_border;
        //     img.fontColour = "#f3f7fb";//"#647382";
        //     img.Text(getX(fraction, text_radius, x_offset-text_x_offset*scale-text_height*0.06),  getY(fraction, text_radius, y_offset+text_y_offset*scale+text_height*0.04),  (text).toString());
        //     img.fontSize=currentFontSize;
        // }
        // img.fontColour = "black";//"#f3f7fb";//"#647382";

        img.Text(
            getX(fraction, text_radius, x_offset - text_x_offset),
            getY(fraction, text_radius, y_offset + text_y_offset),
            text.toString()
        );
    }
}
