/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
overall_rating();

/**
 * Performs the overall rating calculation for the specified models
 */
function overall_rating() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let occupants;

    /* Create a status object to track whether REPORTER Variables are all present and valid.
     * <success> is initially true but will be set to false if anything missing or invalid. */
    let status = { success: true, missing: [], invalid: [] };

    let crash_test_type = get_variable_value(status, `TEMPLATE_CRASH_TEST`, "string");

    if (!status.success) {
        LogWarning(`Variable "TEMPLATE_CRASH_TEST" is missing for overall rating calculation`);
        return;
    }

    if (crash_test_type == "ODB") {
        occupants = ["DRIVER"];
    } else if (crash_test_type == "SOB") {
        occupants = ["DRIVER", "FRONT_PASSENGER"];
    } else if (crash_test_type == "MDB") {
        occupants = ["DRIVER", "REAR_DRIVER_SIDE"];
    } else {
        LogWarning(
            `Variable "TEMPLATE_CRASH_TEST" value "${crash_test_type}" is invalid for overall rating calculation`
        );
        return;
    }

    // Overall rating limits
    let start = get_variable_value(status, `BAND_0_IIHS`, "float");
    let good_upper_limit = get_variable_value(status, `BAND_1_IIHS`, "float");
    let acceptable_upper_limit = get_variable_value(status, `BAND_2_IIHS`, "float");
    let marginal_upper_limit = get_variable_value(status, `BAND_3_IIHS`, "float");
    let poor_upper_limit = get_variable_value(status, `BAND_4_IIHS`, "float");

    for (let m of models) {
        let ratings = {};

        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} overall rating...`);

            /* Get rating for each region and set the weight levels */

            if (crash_test_type == "SOB") {
                ratings[`${occ}_head_and_neck`] = {
                    rating: get_variable_value(status, `${m}_${occ}_HEAD_NECK_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 20
                    }
                };

                ratings[`${occ}_chest`] = {
                    rating: get_variable_value(status, `${m}_${occ}_CHEST_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 20
                    }
                };

                ratings[`${occ}_thigh_and_hip`] = {
                    rating: get_variable_value(status, `${m}_${occ}_THIGH_AND_HIP_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 6,
                        Poor: 10
                    }
                };

                ratings[`${occ}_leg_and_foot`] = {
                    rating: get_variable_value(status, `${m}_${occ}_LEG_OVERALL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 1,
                        Marginal: 2,
                        Poor: 4
                    }
                };

                /* Restraint rating variable name uses "PASSENGER" for the "FRONT_PASSENGER" */
                let occ_alt = occ == "FRONT_PASSENGER" ? "PASSENGER" : occ;

                ratings[`${occ}_restraint`] = {
                    rating: get_variable_value(status, `${m}_${occ_alt}_RESTRAINT_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 6,
                        Poor: 10
                    }
                };

                ratings[`structural`] = {
                    rating: get_variable_value(status, `${m}_OVERALL_STRUCTURAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 6,
                        Poor: 10
                    }
                };
            } else if (crash_test_type == "ODB") {
                ratings[`${occ}_head_and_neck`] = {
                    rating: get_variable_value(status, `${m}_${occ}_HEAD_NECK_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 20
                    }
                };

                ratings[`${occ}_chest`] = {
                    rating: get_variable_value(status, `${m}_${occ}_CHEST_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 20
                    }
                };

                ratings[`${occ}_left_leg`] = {
                    rating: get_variable_value(status, `${m}_${occ}_LEFT_LEG_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 1,
                        Marginal: 2,
                        Poor: 4
                    }
                };

                ratings[`${occ}_right_leg`] = {
                    rating: get_variable_value(status, `${m}_${occ}_RIGHT_LEG_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 1,
                        Marginal: 2,
                        Poor: 4
                    }
                };

                ratings[`structural`] = {
                    rating: get_variable_value(status, `${m}_OVERALL_STRUCTURAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 6,
                        Poor: 10
                    }
                };

                ratings[`restraint`] = {
                    rating: get_variable_value(status, `${m}_RESTRAINT_RATING_DEMERIT`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 1,
                        Marginal: 2,
                        Poor: 4
                    }
                };
            } else if (crash_test_type == "MDB") {
                ratings[`${occ}_head_and_neck`] = {
                    rating: get_variable_value(status, `${m}_${occ}_HEAD_AND_NECK_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 35
                    }
                };

                ratings[`${occ}_torso`] = {
                    rating: get_variable_value(status, `${m}_${occ}_TORSO_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 35
                    }
                };

                ratings[`${occ}_pelvis`] = {
                    rating: get_variable_value(status, `${m}_${occ}_PELVIS_FINAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 6,
                        Poor: 10
                    }
                };

                ratings[`${occ}_head_protection`] = {
                    rating: get_variable_value(status, `${m}_${occ}_HEAD_PROTECTION_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 22
                    }
                };

                ratings[`structural`] = {
                    rating: get_variable_value(status, `${m}_STRUCTURAL_RATING`, "string"),
                    weight: undefined,
                    weight_levels: {
                        Good: 0,
                        Acceptable: 2,
                        Marginal: 10,
                        Poor: 22
                    }
                };
            }

            if (status.success) {
                // Overall weight for each occupant
                if (crash_test_type == "SOB") {
                } else if (crash_test_type == "ODB") {
                } else if (crash_test_type == "MDB") {
                    //Downgrades
                    //Structural downgrade check:
                    /* The vehicle structure rating is intended to limit occupant compartment intrusion at the B-pillar, where
                        occupants of a stature different from the SID-IIs dummies might be sitting. Because good dummy
                        measures cannot indicate general good crashworthiness, a vehicle with a marginal structure rating cannot
                        be rated good overall and can only be rated acceptable with few indicated problems according to the
                        dummy readings.

                        Note that if structural_rating = "Poor" then the weighting from that alone (22) is sufficiently high to mean
                        that the overall weight would be MARGINAL at the very least! For this reason it is commented out below, but left in in
                        case it is required in the future*/
                    /*
                        if ( (structural_rating == "MARGINAL") ) 
                        {
                            if (overall_rating == "GOOD")
                            {
                                overall_rating = "ACCEPTABLE";
                                comment+="GOOD rating downgraded to ACCEPTABLE due to MARGINAL vehicle structure rating.|"; //use | to split comments
                            }     
                        }
                    */
                }
            }
        }

        /* Calculate overall rating for the model */

        /* Set the weight for each body region based on the rating and the weight levels
         * and keep track of the total weight */

        let total_weight = 0;

        for (let region in ratings) {
            let rating = ratings[region].rating;
            let weight_levels = ratings[region].weight_levels;

            ratings[region].weight = weight_levels.Poor; /* Initialise to the worst rating */

            for (let weight_level in weight_levels) {
                if (rating == weight_level) {
                    ratings[region].weight = weight_levels[weight_level];
                    break;
                }
            }

            /* Add the weight for this region to the total weight */

            total_weight += ratings[region].weight;
        }

        /* Calculate the overall rating based on the total weight */

        let overall_test_rating = "Not computed";

        if (total_weight > marginal_upper_limit) {
            overall_test_rating = "Poor";
        } else if (total_weight > acceptable_upper_limit) {
            overall_test_rating = "Marginal";
        } else if (total_weight > good_upper_limit) {
            overall_test_rating = "Acceptable";
        } else {
            overall_test_rating = "Good";
        }

        /* Write variables to the template */

        for (let region in ratings) {
            new Variable(
                templ,
                `${m}_${region.toUpperCase()}_WEIGHT`,
                `${region} weight`,
                ratings[region].weight.toString(),
                "String",
                false,
                true
            );
        }

        /** HG 15th Jan 2025 - A 2024 addition:
         * If a significant fuel leak or compromise of the high-voltage system occurs,
         * both the structural and overall ratings may be downgraded to poor */
        status = { success: true, missing: [], invalid: [] };
        var fuel_voltage_downgrade = get_variable_value(status, `${m}_FUEL_VOLTAGE_DOWNGRADE_DEMERIT`);
        if (status.success && fuel_voltage_downgrade == "Poor") {
            overall_test_rating = "Poor";
            /** KP 28th Jan 2025 - "If the overall is poor the rating dial also should be showing red with the appropriate score for the poor zone.
             * The fuel leak is a significant problem of the structure, even if the occupants rating is good, if it catches fire because of the fuel leak he will die.
             * Just the see the difference between the score and the poor zone, and bring the score upto the poor zone */
            if (total_weight <= marginal_upper_limit) {
                total_weight = marginal_upper_limit + 1;
            }
        }

        new Variable(
            templ,
            `${m}_OVERALL_RATING`,
            "Overall rating",
            overall_test_rating.toString(),
            "String",
            false,
            true
        );

        new Variable(
            templ,
            `${m}_TOTAL_WEIGHT`,
            "Total weight of the model",
            total_weight.toString(),
            "String",
            false,
            true
        );
    }
}
