/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_protection_rating();

/**
 * Performs the head protection rating calculation for the specified models from 2024
 */
function head_protection_rating() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let occupants = ["DRIVER", "REAR_DRIVER_SIDE"];
    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} head protection rating...`);
            let status = { success: true, missing: [], invalid: [] };

            let head_peak_acceleration = get_variable_value(
                status,
                `${m}_${occ}_HEAD_PEAK_ACCELERATION_PEAK_VALUE`,
                "float"
            );

            let head_leaves_vehicle_or_contacts_barrier = false;

            let head_protection_system_is_adequate = WindowAnswerToBool(
                "Head Protection system",
                "Is there an adequate head protection system for the " +
                    `${occ}` +
                    " head?\n(e.g. air bag that deploys properly)"
            );

            let contact_between_head_and_vehicle = WindowAnswerToBool(
                "Contact",
                "Is there contact between the " + `${occ}` + " head and vehicle?"
            );

            let contact_is_hard = false;

            if (contact_between_head_and_vehicle) {
                head_leaves_vehicle_or_contacts_barrier = WindowAnswerToBool(
                    "Head leave vehicle",
                    "Does the " + `${occ}` + " head leave the vehicle or contact the MDB in any way?"
                );

                if (!head_leaves_vehicle_or_contacts_barrier) {
                    contact_is_hard = WindowAnswerToBool(
                        "Hard Contact",
                        "Does the " +
                            `${occ}` +
                            " head have hard contact with interior surfaces or is there bottoming of the side curtain airbag?"
                    );
                }
            }

            let head_protection_rating;

            if (head_leaves_vehicle_or_contacts_barrier) {
                // Poor: Head leaves the vehicle or contacts the moving deformable barrier (MDB)
                head_protection_rating = "Poor";
            } else if (
                !head_protection_system_is_adequate ||
                (head_peak_acceleration > 70 && contact_between_head_and_vehicle)
            ) {
                // Marginal: No head protection system OR contact with interior at >70G
                head_protection_rating = "Marginal";
            } else if (head_protection_system_is_adequate && head_peak_acceleration <= 70 && !contact_is_hard) {
                // Good: Adequate head protection system, acceleration <= 70G, and no hard contact
                head_protection_rating = "Good";
            } else {
                // Acceptable: Adequate head protection system, acceleration <= 70G, but contact between head and vehicle or adequate head protection system, acceleration > 70G and no hard contact
                head_protection_rating = "Acceptable";
            }

            let head_protection_rating_final = new Variable(
                templ,
                `${m}_${occ}_HEAD_PROTECTION_RATING`,
                `${occ} head protection rating`,
                head_protection_rating,
                "String",
                false,
                true
            );

            recalculate_results_IIHS();
        }
    }
}

/**
 * This function is used to create small windows with questions in it. There are two buttons YES or NO. The user can click on either button based on the question.
 * @param {string} heading Heading or title for the window box
 * @param {string} question A question to the user
 * @returns {boolean} returns true or false based on the users input
 */
function WindowAnswerToBool(heading, question) {
    var answer = Window.Message(heading, question, Window.YES | Window.NO);
    return answer === Window.YES;
}
