overall_score();

/**
 * Performs the overall score calculation for the specified models
 */
function overall_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    for (let m of models) {
        LogPrint(`Calculating ${m} overall score...`);

        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };

        /* Body region final scores */
        let scores = get_multiple_variable_values(
            status,
            [
                `${m}_DRIVER_HEAD_FINAL_SCORE`,
                `${m}_DRIVER_CHEST_FINAL_SCORE`,
                `${m}_DRIVER_PELVIS_FINAL_SCORE`,
                `${m}_DRIVER_ABDOMEN_FINAL_SCORE`
            ],
            "float",
            true
        );

        /* Body region capping limits */
        let capping_limits = get_multiple_variable_values(
            status,
            [
                `${m}_DRIVER_HEAD_CAPPING_LIMIT`,
                `${m}_DRIVER_CHEST_CAPPING_LIMIT`,
                `${m}_DRIVER_PELVIS_CAPPING_LIMIT`,
                `${m}_DRIVER_ABDOMEN_CAPPING_LIMIT`
            ],
            "string",
            true
        );

        /* Final scores all start at zero and will remain so if any variables were missing or invalid */
        let overall_score = 0;
        let overall_capping_limit = "*";
        let final_score = 0;

        /* If we have all the required variables, calculate the final scores */
        if (status.success) {
            /* Door opening modifier */
            LogPrint(`Retrieving door opening modifier...`);
            let door_opening_mod = get_variable_value(status, `${m}_DOOR_OPEN_MODIFIER`, "int");
            if (!status.success) {
                warn_about_missing_or_invalid_variables(status, `${m} door opening modifier`);
                door_opening_mod = 0;
            }
            LogPrint(`Retrieving fuel leakage modifier...`);
            let fuel_leakage_mod = get_variable_value(status, `${m}_FUEL_LEAKAGE_MODIFIER`, "int");
            if (!status.success) {
                warn_about_missing_or_invalid_variables(status, `${m} fuel leakage modifier`);
                fuel_leakage_mod = 0;
            }
            /* Overall occupant score is sum of body region scores. */
            overall_score = Object.values(scores).reduce((a, b) => a + b, 0) + door_opening_mod + fuel_leakage_mod;

            LogPrint(`${m} overall score = overall_score`);

            // checks that overall_score is between 0 and 16
            if (overall_score < 0.0) {
                overall_score = 0.0;
            } else if (overall_score > 16.0) {
                overall_score = 16.0;
            }

            /* Only remove the overall occupant capping limit if none of the body region
             * capping limits were exceeded. */
            if (return_capping_limit(Object.values(capping_limits)) == "") {
                overall_capping_limit = "";
                LogPrint(`Capping limit not applied to ${m} DRIVER.`);
                // Display score from 16 points maximum
                final_score = overall_score;
            } else {
                LogPrint(`Capping limit applied to ${m} DRIVER.`);
                final_score = 0;
            }
        } else {
            warn_about_missing_or_invalid_variables(status, `${m} DRIVER overall score calculation`);
        }

        new Variable(templ, `${m}_OVERALL_SCORE`, `Overall score`, overall_score.toString(), "String", false, true);

        new Variable(templ, `${m}_FINAL_SCORE`, `Final score`, final_score.toString(), "String", false, true);

        new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT`,
            `Overall capping limit`,
            overall_capping_limit,
            "String",
            false,
            true
        );

        let capping_limit_explanation = `*Capping limit exceeded`;
        if (overall_capping_limit == "") {
            capping_limit_explanation = "";
        }

        new Variable(
            templ,
            `${m}_OVERALL_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation`,
            capping_limit_explanation,
            "String",
            false,
            true
        );
    }
}
