// module: TRUE

/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
barrier_intrusion_rating();

function barrier_intrusion_rating() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let template_reg = templ.GetVariableValue(`TEMPLATE_REG`);

    for (let m of models) {
        LogPrint(`Calculating ${m} structural rating...`);

        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */

        let status = { success: true, missing: [], invalid: [] };

        let unit_sys_val = get_variable_value(status, `${m}_STRUCTURE_BARRIER_INTRUSION_MODEL_UNITS_VALUE`, "int");

        //=================================================
        //  scoresAndPlots:
        //
        //  Compute compatibility scores and plots:
        //      1) Compute standard deviation, bottoming out and OLC scores
        //      2) Produce the compatibility score graph
        //=================================================

        //========================
        //================ SCORING
        //========================

        let OLC_pct = get_variable_value(status, `${m}_BARRIER_OLC_VELOCITY_OLC_PCT_VALUE`, "float");
        let OLC_score = get_variable_value(
            status,
            `${m}_BARRIER_OLC_VELOCITY_COMPATIBILITY_OCCUPANT_LOAD_CRITERION_SCORE`,
            "float"
        );
        let SD_min_pen = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_STANDARD_DEVIATION_MIN_PENALTY_VALUE`,
            "float"
        );
        let SD_pen_diff = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_STANDARD_DEVIATION_DIFFERENCE_VALUE`,
            "float"
        );
        let SD_max_pen = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_STANDARD_DEVIATION_MAX_PENALTY_VALUE`,
            "float"
        );
        let SD_pct = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_STANDARD_DEVIATION_PERCENTAGE_VALUE`,
            "float"
        );
        let SD_util = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_STANDARD_DEVIATION_UTILISATION_VALUE`,
            "float"
        );
        let bottoming_score = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_BOTTOMING_OUT_SCORE`,
            "float"
        );
        let not_attenuated_score;
        if (template_reg === "C-NCAP") {
            let not_attenuated_val = get_variable_value(
                status,
                `${m}_STRUCTURE_BARRIER_INTRUSION_LONGITUDINAL_INTRUSION_ATTENUATION_VALUE`,
                "float"
            );
            not_attenuated_score = get_variable_value(
                status,
                `${m}_STRUCTURE_BARRIER_INTRUSION_LONGITUDINAL_INTRUSION_ATTENUATION_SCORE`,
                "float"
            );

            let not_attenuated_boolean = not_attenuated_val === 1 ? "Yes" : "No";
            new Variable(
                templ,
                `${m}_LONGITUDINAL_INTRUSION_ATTENUATION_BOOLEAN`,
                `Barrier height intrusion boolean (Yes/No)`,
                not_attenuated_boolean,
                "String",
                false,
                true
            );
        }

        // Find max possible SD penalty depending on OLC % -- range_SD is already scaled
        let range_SD;
        if (OLC_pct <= 0) {
            range_SD = SD_min_pen;
        } else if (OLC_pct > 0 && OLC_pct <= 1) {
            range_SD = SD_pen_diff * OLC_pct + SD_min_pen; // i.e. interpolate between extremes when OLC=0% and OLC=100%
        } else {
            range_SD = SD_max_pen;
        }

        let SD_score;
        if (SD_pct < 0) {
            // i.e. SD < SD_scale_min
            SD_pct = 0.0;
            SD_score = 0;
            SD_util = 0.0; // bound the SD_util between 0% and 100%
        } else if (SD_pct >= 0 && SD_pct <= 1) {
            SD_score = SD_pct * range_SD; // scaled
        } // i.e. SD > SD_scale_max; full penalty
        else {
            SD_pct = 1.0;
            SD_score = range_SD; // scaled
            SD_util = 100.0;
        }

        let unit_system;
        switch (unit_sys_val) {
            case 1:
                unit_system = "m, kg, s";
                break;
            case 2:
                unit_system = "mm, t, s";
                break;
            case 3:
                unit_system = "mm, kg, ms";
                break;
            case 4:
                unit_system = "mm, g, ms";
                break;
            case 5:
                unit_system = "ft, slug, ms";
                break;
            case 6:
                unit_system = "m, t, s";
                break;
            default:
                unit_system = "Not defined";
                break;
        }
        new Variable(templ, `${m}_MODEL_UNITS`, `Model unit system`, unit_system, "String", false, true);

        let bottoming_out_boolean = bottoming_score === 0 ? "No" : "Yes";
        new Variable(
            templ,
            `${m}_BOTTOMING_OUT_BOOLEAN`,
            `Barrier bottoming out boolean (Yes/No)`,
            bottoming_out_boolean,
            "String",
            false,
            true
        );

        new Variable(
            templ,
            `${m}_STANDARD_DEVIATION_MAX_POSS_SCORE`,
            `Maximum score for the SD for the current OLC range`,
            range_SD.toString(),
            "String",
            false,
            true
        );

        new Variable(
            templ,
            `${m}_COMPATIBILITY_STANDARD_DEVIATION_SCORE`,
            `Compatibility SD score`,
            SD_score.toString(),
            "String",
            false,
            true
        );

        let max_compatibility_penalty = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_MAX_COMPATIBILITY_PENALTY_VALUE`,
            "int"
        );

        if (status.success) {
            let additional_penalty;
            additional_penalty = template_reg === "C-NCAP" ? bottoming_score + not_attenuated_score : bottoming_score;
            let compatibility_score = Math.max(max_compatibility_penalty, OLC_score + SD_score + additional_penalty);
            new Variable(
                templ,
                `${m}_COMPATIBILITY_TOTAL_SCORE`,
                `Compatibility assessment penalty`,
                compatibility_score.toString(),
                "String",
                false,
                true
            );
        } else {
            warn_about_missing_or_invalid_variables(status, `${m} compatibility calculation`);
        }
    }
}
