/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
draw_structural_graphs();

/**
 * This script updates the format:cell colour and border thicknesses
 * of the Reporter table so to reflect the MPDB intrusion contour plot
 *
 * Get the currect template and the relevant variables
 *
 */
//
//

//
// to produce the intrusion contour plot table

/**
 * Draw structural graphs for the specified occupants
 */
function draw_structural_graphs() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);

    let table_1_page_num = 0;
    let table_2_page_num = 5;

    /* Create a status object to track whether REPORTER Variables are all present and valid.
     * <success> is initially true but will be set to false if anything missing or invalid. */
    let status = { success: true, missing: [], invalid: [] };

    is_valid_output_dir(output_dir);

    if (!status.success) {
        LogWarning(`Variable "TEMPLATE_CRASH_TEST" is missing for draw image `);
        return;
    }

    for (let m of models) {
        /////////////////////////////////////// Draw barrier intrusion depth graph ///////////////////////////////////////
        // Fetch necessary variables
        let driver_side_val = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_LEFT_HAND_DRIVE_VALUE`,
            "int"
        );
        let driver_side = driver_side_val === 1 ? "left" : "right";

        new Variable(
            templ,
            `${m}_DRIVE_SIDE`,
            `Drive side (LEFT/RIGHT)`,
            driver_side.toUpperCase(),
            "String",
            false,
            true
        );

        let imp_width = get_variable_value(status, `${m}_STRUCTURE_BARRIER_INTRUSION_VEHICLE_WIDTH_VALUE`, "float");

        let results_scheme_value = get_variable_value(
            status,
            `${m}_STRUCTURE_BARRIER_INTRUSION_EURONCAP_COLOURS_VALUE`,
            "int"
        );
        let results_scheme = results_scheme_value === 1 ? "SPREADSHEET" : "DEFAULT";

        /**
         * The euroncap.com Compatibility Assessment Spreadsheet has an unusual contour band
         * colour scheme for the barrier deformation table:
         *
         * https://cdn.euroncap.com/media/42267/euro-ncap-spreadsheet-v202001-mpdb-only.201811211115080956.xlsm
         *
         * Rather than the green-yellow-orange-brown-red scheme used elsewhere in the protocols
         * (indicating good-adequate-marginal-weak-poor), it uses an orange-yellow-green-red-brown
         * scheme. Apparently, the rationale is that because the assessment is based on a standard
         * deviation, values with medium deformations are likely to be better. Regardless of whether
         * you agree with those assumptions, some users find it convenient to use a colour scheme
         * consistent with the spreadsheet, so we provide two options:
         *
         * "DEFAULT": matches the protocol colour scheme, more typical sequence of colours
         * "SPREADSHEET": matches the EuroNCAP spreadsheet scheme
         */

        /* Colour schemes for EuroNCAP protocol and spreadsheet */
        const EuroNCAPColour = {
            // Standard EuroNCAP protocol colour scheme (DEFAULT)
            GREEN: Colour.RGB(38, 155, 41),
            YELLOW: Colour.RGB(255, 204, 0),
            ORANGE: Colour.RGB(255, 151, 0),
            BROWN: Colour.RGB(117, 63, 42),
            RED: Colour.RGB(224, 7, 0)
        };

        const SpreadsheetColour = {
            // Compatibility Assessment Spreadsheet colour scheme (SPREADSHEET)
            GREEN: Colour.RGB(146, 208, 80),
            YELLOW: Colour.RGB(255, 255, 0),
            ORANGE: Colour.RGB(255, 192, 0),
            BROWN: Colour.RGB(192, 0, 0),
            RED: Colour.RGB(255, 0, 0)
        };

        let contour_bands = [];
        if (results_scheme === "SPREADSHEET") {
            // euroncap.com spreadsheet colour scheme
            contour_bands = [
                { colour: SpreadsheetColour.ORANGE, value: 160 },
                { colour: SpreadsheetColour.YELLOW, value: 320 },
                { colour: SpreadsheetColour.GREEN, value: 480 },
                { colour: SpreadsheetColour.RED, value: 640 },
                { colour: SpreadsheetColour.BROWN, value: 1e8 }
            ];
        } else {
            // default colour scheme
            contour_bands = [
                { colour: EuroNCAPColour.GREEN, value: 160 },
                { colour: EuroNCAPColour.YELLOW, value: 320 },
                { colour: EuroNCAPColour.ORANGE, value: 480 },
                { colour: EuroNCAPColour.BROWN, value: 630 },
                { colour: EuroNCAPColour.RED, value: 1e8 }
            ];
        }

        // Initialise the intrusion vector and read the MPDB_barrier_def.csv file.
        let intrusion = []; // from D3PLOT
        let filename = `${output_dir}/MPDB_barrier_def.csv`;
        if (!File.Exists(filename)) {
            LogError("Cannot find the intrusion displacements list: " + filename);
            Exit();
        }

        let f_csv = new File(filename, File.READ);
        /* MPDB_barrier_def.csv is in the form:
         **
         ** || x-coordinate
         ** || Coordinate (x)
         ** || 251.82416318447088
         ** || ...
         **
         ** So we need to skip the two header lines */
        f_csv.ReadLine();
        f_csv.ReadLine();
        intrusion = new Array(1400);
        let line;
        let i = 0;

        while ((line = f_csv.ReadLine()) !== undefined && i < 1400) {
            intrusion[i] = parseFloat(line);
            i++;
        }

        f_csv.Close();

        // The deformation table is shown on the summary table without numerical values,
        // and in a larger size on the final page, with numerical values
        let tables = [];
        let p = templ.GetPage(table_1_page_num);
        tables.push(Item.GetFromName(p, "Intrusion_Table_1"));
        p = templ.GetPage(table_2_page_num);
        tables.push(Item.GetFromName(p, "Intrusion_Table_2"));

        // Update intrusion table colours and borders
        for (let i = 0; i < intrusion.length; i++) {
            // Get the z and  y coordinatein the local barrier reference system
            let z_coord = 560 - getZ(i);
            let y_coord = getY(i);

            // Get the respective row and column numbers
            let row = 27 - Math.floor((z_coord - 10) / 20);
            let column = Math.floor((y_coord - 10) / 20);

            // Initialize the cell border to a 0.1 width
            let cell_border = { rightBorderWidth: 0.1, topBorderWidth: 0.1 };
            SetCellPropertiesForTables(cell_border, row, column);

            // Set the cell border thickness to a larger value: 1.0, if the table cell
            // is at the boundary of the asssessment area
            if (checkLeftBorder(intrusion[i], z_coord, column, driver_side, imp_width)) {
                cell_border.leftBorderWidth = 1.0;
                SetCellPropertiesForTables(cell_border, row, column);
            }

            if (checkRightBorder(intrusion[i], z_coord, column, driver_side, imp_width)) {
                cell_border.rightBorderWidth = 1.0;
                SetCellPropertiesForTables(cell_border, row, column);
            }

            if (checkTopBorder(intrusion[i], z_coord, column, driver_side, imp_width)) {
                cell_border.topBorderWidth = 1.0;
                SetCellPropertiesForTables(cell_border, row, column);
            }

            if (checkBotBorder(intrusion[i], z_coord, column, driver_side, imp_width)) {
                cell_border.bottomBorderWidth = 1.0;
                SetCellPropertiesForTables(cell_border, row, column);
            }

            // Per each cell assign a different colour based on the value of intrusion
            let cell_colour = {};
            for (let j = 0; j < contour_bands.length; j++) {
                if (intrusion[i] < contour_bands[j].value) {
                    cell_colour.fillColour = contour_bands[j].colour;
                    break;
                }
            }

            row = 27 - row;
            SetCellPropertiesForTables(cell_colour, row, column);

            // For the bigger table on the final page, include the values
            let cell_text = { text: Math.round(intrusion[i]).toString() };
            tables[1].SetCellProperties(cell_text, row, column);
        }

        // The tables have contour bar legends (with differing designs) that
        // need to be updated depending on the selected colour scheme
        let legends = [];
        p = templ.GetPage(table_1_page_num);
        legends.push(Item.GetFromName(p, "Barrier_intrusion_contour_bar_1"));
        p = templ.GetPage(table_2_page_num);
        legends.push(Item.GetFromName(p, "Barrier_intrusion_contour_bar_2"));

        for (let j = 0; j < contour_bands.length; j++) {
            let colour = { fillColour: contour_bands[j].colour };
            let text = { text: contour_bands[j].value.toString() };

            // Contour bar legend on final page (single column no units)
            legends[1].SetCellProperties(colour, 2 * j, 0);
            if (j < contour_bands.length - 1) {
                // No upper value associated with final colour
                legends[1].SetCellProperties(text, 2 * j + 1, 0);
            }

            // Contour bar on summary page (two columns, mm units)
            text.text += " mm";
            legends[0].SetCellProperties(colour, 2 * j, 1);
            if (j < contour_bands.length - 1) {
                legends[0].SetCellProperties(text, 2 * j + 1, 0);
            }
        }

        /**
         * Sets the cell properties for all tables in the global array <tables>.
         * @param {Object} obj The properties object to apply to the cells
         * @param {number} row Row index of the cell
         * @param {number} column Column index of the cell
         */
        function SetCellPropertiesForTables(obj, row, column) {
            // Sets the cell properties for all of the tables in global array <tables>
            for (var i = 0; i < tables.length; i++) {
                tables[i].SetCellProperties(obj, row, column);
            }
        }
    }
}

/**
 * Checks if the left border of a cell should be thickened based on its position.
 * @param {number} z_coord z coordinate of the cell
 * @param {number} column Column index of the cell
 * @param {string} driver_side Side of the driver (left or right)
 * @param {number} imp_width Width of the vehicle
 * @returns {boolean} True if the left border should be thickened, false otherwise
 */
function checkLeftBorder(intrusion, z_coord, column, driver_side, imp_width) {
    let assess_flag = false;
    const eff_imp_width = 0.45 * imp_width;

    if (z_coord < 500 && z_coord > 100) {
        let col_border;

        if (driver_side === "left") {
            col_border = Math.ceil((1000 - eff_imp_width) / 20);

            col_border = Math.max(0, Math.min(col_border, 49));

            if (column === col_border) {
                assess_flag = true;
            }
        } else {
            col_border = 10;

            if (column === col_border) {
                assess_flag = true;
            }
        }
    }

    return assess_flag;
}

/**
 * Checks if the right border of a cell should be thickened based on its position.
 * @param {number} z_coord z coordinate of the cell
 * @param {number} column Column index of the cell
 * @param {string} driver_side Side of the driver (left or right)
 * @param {number} imp_width Width of the vehicle
 * @returns {boolean} True if the right border should be thickened, false otherwise
 */
function checkRightBorder(intrusion, z_coord, column, driver_side, imp_width) {
    let assess_flag = false;
    const eff_imp_width = 0.45 * imp_width;

    if (z_coord < 500 && z_coord > 100) {
        let col_border;

        if (driver_side === "right") {
            col_border = Math.floor(eff_imp_width / 20) - 1;

            col_border = Math.max(0, Math.min(col_border, 49));

            if (column === col_border) {
                assess_flag = true;
            }
        } else {
            col_border = 39;

            if (column === col_border) {
                assess_flag = true;
            }
        }
    }

    return assess_flag;
}

/**
 * Checks if the top border of a cell should be thickened based on its position.
 * @param {number} z_coord z coordinate of the cell
 * @param {number} column Column index of the cell
 * @param {string} driver_side Side of the driver (left or right)
 * @param {number} imp_width Width of the vehicle
 * @returns {boolean} True if the top border should be thickened, false otherwise
 */
function checkTopBorder(intrusion, z_coord, column, driver_side, imp_width) {
    let assess_flag = false;
    let eff_imp_width = 0.45 * imp_width;

    if (z_coord === 490) {
        if (driver_side === "left") {
            if (column >= Math.ceil((1000 - eff_imp_width) / 20) && column <= 39) {
                assess_flag = true;
            }
        } else {
            if (column >= 10 && column <= Math.floor(eff_imp_width / 20) - 1) {
                assess_flag = true;
            }
        }
    }

    return assess_flag;
}

/**
 * Checks if the bottom border of a cell should be thickened based on its position.
 * @param {number} z_coord z coordinate of the cell
 * @param {number} column Column index of the cell
 * @param {string} driver_side Side of the driver (left or right)
 * @param {number} imp_width Width of the vehicle
 * @returns {boolean} True if the bottom border should be thickened, false otherwise
 */
function checkBotBorder(intrusion, z_coord, column, driver_side, imp_width) {
    let assess_flag = false;
    let eff_imp_width = 0.45 * imp_width;

    if (z_coord == 110) {
        if (driver_side === "left") {
            if (column >= Math.ceil((1000 - eff_imp_width) / 20) && column <= 39) {
                assess_flag = true;
            }
        } else {
            if (column >= 10 && column <= Math.floor(eff_imp_width / 20) - 1) {
                assess_flag = true;
            }
        }
    }

    return assess_flag;
}

/**
 * Converts the z-axis index to the actual z-coordinate value in millimeters.
 * @param {number} i z-axis index
 * @returns {number} Corresponding z-coordinate value in millimeters
 */
function getZ(i) {
    return 550 - Math.floor((i % 28) * 20);
}

/**
 * Converts the y-axis index to the actual y-coordinate value in millimeters.
 * @param {number} i y-axis index
 * @returns {number} Corresponding y-coordinate value in millimeters
 */
function getY(i) {
    return Math.floor(i / 28 + 1) * 20 - 10;
}
