/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
knee_femur_and_pelvis_score();

/**
 * Performs the knee, femur and pelvis score calculation for the specified models
 */
function knee_femur_and_pelvis_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `knee, femur and pelvis`;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    let sides = ["LEFT", "RIGHT"];
    let side_scores = {}; // To be populated with score for each side
    for (let m of models) {
        for (let occ of occupants) {
            for (let side of sides) {
                LogPrint(`Calculating ${m} ${occ} ${side} ${body_region_label} score...`);

                /* Create a status object to track whether REPORTER Variables are all present and valid.
                 * <success> is initially true but will be set to false if anything missing or invalid. */
                let status = { success: true, missing: [], invalid: [] };

                /* Knee, femur and pelvis score and modifiers */

                /* Driver */
                let acetabular_score;

                /* Shared */
                let femur_score;
                let knee_score;
                let knee_variable_contact_mod;
                let knee_concentrated_load_mod;
                let knee_airbag_deployment_mod;

                switch (occ) {
                    case "DRIVER":
                        acetabular_score = get_variable_value(
                            status,
                            `${m}_${occ}_ACETABULAR_FORCE_${side}_SCORE`,
                            "float"
                        );
                        break;
                    case "FRONT_PASSENGER":
                        /* No variables specific to front passenger */
                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }

                /* Variables extracted for all occupants */
                femur_score = get_variable_value(
                    status,
                    `${m}_${occ}_${side}_FEMUR_COMPRESSION_EXCEEDENCE_SCORE`,
                    "float"
                );
                knee_score = get_variable_value(status, `${m}_${occ}_${side}_KNEE_COMPRESSION_SCORE`, "float");
                knee_variable_contact_mod = get_variable_value(
                    status,
                    `${m}_${occ}_${side}_KNEE_VARIABLE_CONTACT_MODIFIER`,
                    "float"
                );
                knee_concentrated_load_mod = get_variable_value(
                    status,
                    `${m}_${occ}_${side}_KNEE_CONCENTRATED_LOADING_MODIFIER`,
                    "float"
                );
                knee_airbag_deployment_mod = get_variable_value(
                    status,
                    `${m}_${occ}_${side}_KNEE_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`,
                    "float"
                );

                /* Final scores all start at zero and will remain so if any variables were missing or invalid */
                side_scores[side] = 0;

                /* If we have all the required variables, calculate the final scores */
                if (status.success) {
                    /* Score calculation is different for different occupants */
                    switch (occ) {
                        case "DRIVER":
                            /* Left/right score is minimum of acetabular, femur and knee scores, summed with modifiers */
                            side_scores[side] = Math.max(
                                0,
                                Math.min(acetabular_score, femur_score, knee_score) +
                                    knee_variable_contact_mod +
                                    knee_concentrated_load_mod +
                                    knee_airbag_deployment_mod
                            );
                            break;
                        case "FRONT_PASSENGER":
                            /* Left/right score is minimum of femur and knee scores, summed with modifiers */
                            side_scores[side] = Math.max(
                                0,
                                Math.min(femur_score, knee_score) +
                                    knee_variable_contact_mod +
                                    knee_concentrated_load_mod +
                                    knee_airbag_deployment_mod
                            );
                            break;
                        default:
                            LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                            Exit();
                    }

                    LogPrint(`${m} ${occ} ${side} ${body_region_label} score = ${side_scores[side]}`);
                } else {
                    warn_about_missing_or_invalid_variables(
                        status,
                        `${m} ${occ} ${body_region_label} score calculation`
                    );
                }

                let side_score_var = new Variable(
                    templ,
                    `${m}_${occ}_KNEE_FEMUR_PELVIS_${side}_FINAL_SCORE`,
                    `Final ${side} ${body_region_label} score`,
                    side_scores[side].toString(),
                    "String",
                    false,
                    true
                );
            }

            /* Overall score for each occupant is minimum of left and right scores */
            let knee_femur_pelvis_final_score = Math.min(side_scores.LEFT, side_scores.RIGHT);
            let knee_femur_pelvis_final_score_var = new Variable(
                templ,
                `${m}_${occ}_KNEE_FEMUR_PELVIS_FINAL_SCORE`,
                `Final ${body_region_label} score`,
                knee_femur_pelvis_final_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
