/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest score calculation for the specified models
 */
function chest_score() {
    let template = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = "chest";

    let occupants = ["DRIVER", "REAR_PASSENGER_SIDE"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            let chest_compression_score = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_SCORE`,
                "float",
                template
            );

            let chest_viscous_criterion_score = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_CHEST_VISCOUS_CRITERION_SCORE`,
                "float",
                template
            );
            let chest_compression_score_banded = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_SCORE_ENCAP_BANDED`,
                "float"
            );

            let chest_viscous_criterion_score_banded = get_variable_value(
                status,
                `${m}_${occ}_CHEST_VISCOUS_CRITERION_SCORE_ENCAP_BANDED`,
                "float"
            );

            let steering_wheel_contact_mod = getModifierValue(
                status,
                `${m}_${occ}_CHEST_STEERING_WHEEL_CONTACT_MODIFIER`
            );
            let shoulder_belt_load_mod = getModifierValue(status, `${m}_${occ}_CHEST_BELT_LOAD_MODIFIER`);
            let incorrect_airbag_mod = getModifierValue(
                status,
                `${m}_${occ}_CHEST_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`
            );

            let modifier_total = steering_wheel_contact_mod + shoulder_belt_load_mod + incorrect_airbag_mod;
            let modifier_total_pass = shoulder_belt_load_mod + incorrect_airbag_mod;
            let chest_and_abdomen_final_nomod;

            const cc = chest_compression_score_banded;
            const cv = chest_viscous_criterion_score_banded;
            const mod = modifier_total;
            const modp = modifier_total_pass;

            let chest_and_abdomen_final = Math.min(cc, cv) + mod;
            let chest_and_abdomen_pass_final = Math.min(cc, cv) + modp;
            chest_and_abdomen_final_nomod = Math.min(cc, cv);

            /* Get the head excursion zone and whether there's a countermeasure or not - required to set the high score value */
            let head_excursion_zone = get_variable_value(status, `${m}_STRUCTURE_HEAD_EXCURSION_RATING`, "string");
            let countermeasure_int = get_variable_value(
                status,
                `${m}_STRUCTURE_HEAD_EXCURSION_COUNTERMEASURE_VALUE`,
                "int"
            );

            /* comment why checking for countermeasure_int === 1 */
            let countermeasure = countermeasure_int === 1;

            let CHEST_HI_SCORE = get_max_chest_score(head_excursion_zone, countermeasure);

            new_variable_and_new_euroNCAP_banded_score(
                template,
                `CHEST_HI_SCORE`,
                `Max chest score for head excursion in ${head_excursion_zone} zone with${
                    countermeasure ? "" : "out"
                } countermeasure.`,
                CHEST_HI_SCORE.toString(),
                "General",
                false,
                true
            );

            let CHEST_COMPRESSION_HI_LIMIT = 28; // The compression limit for the chest in mm
            let CHEST_COMPRESSION_LO_LIMIT = 50; // The compression limit for the chest in mm
            let CHEST_COMPRESSION_CAPPING_LIMIT = 50; // Capping limit is 50mm (same as lower limit)

            //set perfomance limits
            let CHEST_COMPRESSION_GOOD = CHEST_COMPRESSION_HI_LIMIT; //higher performance limit
            let CHEST_COMPRESSION_WEAK = CHEST_COMPRESSION_LO_LIMIT; //lower performance limit
            let CHEST_TOP_HI_SCORE = CHEST_HI_SCORE;
            let CHEST_TOP_LO_SCORE = 0;
            let CHEST_MID_HI_SCORE = CHEST_HI_SCORE;
            let CHEST_MID_LO_SCORE = 0;
            let CHEST_BOTTOM_HI_SCORE = CHEST_HI_SCORE;
            let CHEST_BOTTOM_LO_SCORE = 0;
            let max_upp_rib_com = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_MAX_01_VALUE`, "float");
            let max_mid_rib_com = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_MAX_02_VALUE`, "float");
            let max_bottom_rib_com = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_MAX_03_VALUE`, "float");

            if (max_upp_rib_com == "Missing" || max_upp_rib_com == null) {
                var upper_rib_compression_score = 0;
            } else {
                var upper_rib_compression_score = sliding_scale(
                    max_upp_rib_com,
                    CHEST_COMPRESSION_GOOD,
                    CHEST_COMPRESSION_WEAK,
                    CHEST_TOP_HI_SCORE,
                    CHEST_TOP_LO_SCORE
                );
            }

            if (max_mid_rib_com == "Missing" || max_mid_rib_com == null) {
                var mid_rib_compression_score = 0;
            } else {
                var mid_rib_compression_score = sliding_scale(
                    max_mid_rib_com,
                    CHEST_COMPRESSION_GOOD,
                    CHEST_COMPRESSION_WEAK,
                    CHEST_MID_HI_SCORE,
                    CHEST_MID_LO_SCORE
                );
            }

            if (max_bottom_rib_com == "Missing" || max_bottom_rib_com == null) {
                var bottom_rib_compression_score = 0;
            } else {
                var bottom_rib_compression_score = sliding_scale(
                    max_bottom_rib_com,
                    CHEST_COMPRESSION_GOOD,
                    CHEST_COMPRESSION_WEAK,
                    CHEST_BOTTOM_HI_SCORE,
                    CHEST_BOTTOM_LO_SCORE
                );
            }

            new Variable(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_01_SCORE`,
                `Result from Automotive Assessments Workflow`,
                upper_rib_compression_score.toFixed(3),
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_02_SCORE`,
                `Result from Automotive Assessments Workflow`,
                mid_rib_compression_score.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_03_SCORE`,
                `Result from Automotive Assessments Workflow`,
                bottom_rib_compression_score.toFixed(3),
                "General",
                false,
                true
            );
            let maxc = Math.max(max_upp_rib_com, max_mid_rib_com, max_bottom_rib_com);
            new Variable(
                template,
                `CHEST_MAX_COMPRESSION`,
                `Result from Automotive Assessments Workflow`,
                maxc.toFixed(3),
                "General",
                false,
                true
            );
            let chest_score = Math.min(
                upper_rib_compression_score,
                mid_rib_compression_score,
                bottom_rib_compression_score
            );
            new Variable(
                template,
                `CHEST_MAX_COMPRESSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                chest_score.toFixed(3),
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_CHEST_CAPPING_LIMIT`,
                `Result from Automotive Assessments Workflow`,
                maxc >= CHEST_COMPRESSION_CAPPING_LIMIT ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );
            let chest_and_abdomen_final_score;
            let chest_and_abdomen_final_points;

            chest_and_abdomen_final_score = Math.max(Math.min(chest_and_abdomen_final, 100.0), 0.0);
            chest_and_abdomen_final_points = (chest_and_abdomen_final_score / 100) * 1.25;

            const fallback = (x) => (isNaN(x) ? "Missing" : Math.max(Math.min(x, 100), 0));

            chest_and_abdomen_final_score = fallback(chest_and_abdomen_final_score);
            chest_and_abdomen_final_points = isNaN(chest_and_abdomen_final_points)
                ? "Missing"
                : chest_and_abdomen_final_points;

            let chest_and_abdomen_pass_final_score;
            let chest_and_abdomen_pass_final_points;

            chest_and_abdomen_pass_final_score = Math.max(Math.min(chest_and_abdomen_pass_final, 100.0), 0.0);
            chest_and_abdomen_pass_final_points = (chest_and_abdomen_pass_final_score / 100) * 1.25;

            chest_and_abdomen_pass_final_score = fallback(chest_and_abdomen_pass_final_score);
            chest_and_abdomen_pass_final_points = isNaN(chest_and_abdomen_pass_final_points)
                ? "Missing"
                : chest_and_abdomen_pass_final_points;
            chest_and_abdomen_final_nomod = isNaN(chest_and_abdomen_final_nomod)
                ? "Missing"
                : chest_and_abdomen_final_nomod;

            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_FINAL_SCORE`,
                "Final chest score",
                chest_and_abdomen_final_score.toString(),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_FINAL_POINTS`,
                "Final chest score",
                chest_and_abdomen_final_points.toString(),
                "General",
                false,
                true
            );

            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_FINAL_PASSENGER_SCORE`,
                "Final chest score",
                chest_and_abdomen_pass_final_score.toString(),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_FINAL_PASSENGER_POINTS`,
                "Final chest score",
                chest_and_abdomen_pass_final_points.toString(),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_FINAL_SCORE_NOMOD`,
                "Final chest score no modifier",
                chest_and_abdomen_final_nomod.toString(),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_CHEST_MODIFIER_TOTAL`,
                `Final ${body_region_label} modifier`,
                modifier_total.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_CHEST_MODIFIER_PASSENGER_TOTAL`,
                `Final ${body_region_label} modifier`,
                modifier_total_pass.toFixed(3),
                "General",
                false,
                true
            );
        }
    }
}

/**
 * Returns the maximum chest score based on the head excursion zone and countermeasure
 * @param {string} zone The head excursion zone
 * @param {boolean} countermeasure Whether there is a countermeasure or not
 * @returns {number} The maximum chest score
 */
function get_max_chest_score(zone, countermeasure) {
    if (countermeasure) {
        switch (zone) {
            case "CAPPING":
            case "RED":
                return 0;
            case "BROWN":
                return 2;
            case "ORANGE":
                return 3;
            case "YELLOW":
                return 4;
            case "GREEN":
                return 4;
            default:
                return 0;
        }
    } else {
        switch (zone) {
            case "CAPPING":
            case "RED":
                return 0;
            case "ORANGE":
                return 1;
            case "YELLOW":
                return 2;
            case "GREEN":
                return 4;
            default:
                return 0;
        }
    }
}

function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
