/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
abdomen_score();

/**
 * Performs the abdomen score calculation for the specified models
 */
function abdomen_score() {
    let template = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = "abdomen";

    let occupants = ["FRONT_PASSENGER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* ABDOMEN_HI_SCORE copies chest */
            let ABDOMEN_HI_SCORE = get_variable_value(status, `CHEST_HI_SCORE`, "int");

            let ABDOMEN_COMPRESSION_HI_LIMIT = 88; // The abdominal compression limit in mm
            let ABDOMEN_COMPRESSION_LO_LIMIT = 88; // The abdominal compression limit in mm
            let ABDOMEN_COMPRESSION_CAPPING_LIMIT = 88; // Capping limit is 65mm (same as lower limit)
            let ABDOMEN_COMPRESSION_GOOD = ABDOMEN_COMPRESSION_HI_LIMIT; //higher performance limit
            let ABDOMEN_COMPRESSION_WEAK = ABDOMEN_COMPRESSION_LO_LIMIT; //lower performance limit

            let max_upp_rib_com = get_variable_value(status, `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_01_VALUE`, "float");
            let max_bottom_rib_com = get_variable_value(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_02_VALUE`,
                "float"
            );
            let shoulder_belt_load_mod = getModifierValue(status, `${m}_${occ}_CHEST_BELT_LOAD_MODIFIER`);
            let incorrect_airbag_mod = getModifierValue(
                status,
                `${m}_${occ}_CHEST_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`
            );
            let modifier_total = shoulder_belt_load_mod + incorrect_airbag_mod;

            let CHEST_COMPRESSION_UPPER_LEFT = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LEFT_UPPER_VALUE`,
                "float"
            );
            let CHEST_COMPRESSION_LOWER_LEFT = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LEFT_LOWER_VALUE`,
                "float"
            );
            let CHEST_COMPRESSION_UPPER_RIGHT = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_RIGHT_UPPER_VALUE`,
                "float"
            );
            let CHEST_COMPRESSION_LOWER_RIGHT = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_RIGHT_LOWER_VALUE`,
                "float"
            );
            let CHEST_COMPRESSION_UPPER_LEFT_SCORE = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LEFT_UPPER_SCORE`,
                "float",
                template
            );
            let CHEST_COMPRESSION_LOWER_LEFT_SCORE = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LEFT_LOWER_SCORE`,
                "float",
                template
            );
            let CHEST_COMPRESSION_UPPER_RIGHT_SCORE = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_RIGHT_UPPER_SCORE`,
                "float",
                template
            );
            let CHEST_COMPRESSION_LOWER_RIGHT_SCORE = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_RIGHT_LOWER_SCORE`,
                "float",
                template
            );
            let CHEST_COMPRESSION_UPPER_LEFT_SCORE_BANDED = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LEFT_UPPER_SCORE_ENCAP_BANDED`,
                "float"
            );
            let CHEST_COMPRESSION_LOWER_LEFT_SCORE_BANDED = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LEFT_LOWER_SCORE_ENCAP_BANDED`,
                "float"
            );
            let CHEST_COMPRESSION_UPPER_RIGHT_SCORE_BANDED = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_RIGHT_UPPER_SCORE_ENCAP_BANDED`,
                "float"
            );
            let CHEST_COMPRESSION_LOWER_RIGHT_SCORE_BANDED = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_RIGHT_LOWER_SCORE_ENCAP_BANDED`,
                "float"
            );
            let ABDOMEN_COMPRESSION_LEFT = get_variable_value(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_LEFT_MAX_VALUE`,
                "float"
            );
            let ABDOMEN_COMPRESSION_RIGHT = get_variable_value(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_RIGHT_MAX_VALUE`,
                "float"
            );
            let ABDOMEN_COMPRESSION_LEFT_SCORE = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_LEFT_SCORE`,
                "float",
                template
            );
            let ABDOMEN_COMPRESSION_RIGHT_SCORE = get_variable_value_and_ENCAP_band_score(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_RIGHT_SCORE`,
                "float",
                template
            );
            let ABDOMEN_COMPRESSION_LEFT_SCORE_BANDED = get_variable_value(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_LEFT_SCORE_ENCAP_BANDED`,
                "float"
            );
            let ABDOMEN_COMPRESSION_RIGHT_SCORE_BANDED = get_variable_value(
                status,
                `${m}_${occ}_ABDOMEN_COMPRESSION_RIGHT_SCORE_ENCAP_BANDED`,
                "float"
            );

            let CHEST_COMPRESSION_MAX_VALUE;
            let CHEST_COMPRESSION_SCORE;
            let ABDOMEN_COMPRESSION_MAX_VALUE;
            let ABDOMEN_COMPRESSION_SCORE;

            const isInvalid = (val) => val === "Missing" || isNaN(val);

            // Chest Compression

            const chestValues = [
                CHEST_COMPRESSION_UPPER_LEFT,
                CHEST_COMPRESSION_LOWER_LEFT,
                CHEST_COMPRESSION_UPPER_RIGHT,
                CHEST_COMPRESSION_LOWER_RIGHT,
                CHEST_COMPRESSION_UPPER_LEFT_SCORE_BANDED,
                CHEST_COMPRESSION_LOWER_LEFT_SCORE_BANDED,
                CHEST_COMPRESSION_UPPER_RIGHT_SCORE_BANDED,
                CHEST_COMPRESSION_LOWER_RIGHT_SCORE_BANDED
            ];

            if (chestValues.some(isInvalid)) {
                CHEST_COMPRESSION_MAX_VALUE = "Missing";
                CHEST_COMPRESSION_SCORE = "Missing";
            } else {
                CHEST_COMPRESSION_MAX_VALUE = Math.max(
                    ...[
                        CHEST_COMPRESSION_UPPER_LEFT,
                        CHEST_COMPRESSION_LOWER_LEFT,
                        CHEST_COMPRESSION_UPPER_RIGHT,
                        CHEST_COMPRESSION_LOWER_RIGHT
                    ].map(Number)
                );

                CHEST_COMPRESSION_SCORE = Math.min(
                    ...[
                        CHEST_COMPRESSION_UPPER_LEFT_SCORE_BANDED,
                        CHEST_COMPRESSION_LOWER_LEFT_SCORE_BANDED,
                        CHEST_COMPRESSION_UPPER_RIGHT_SCORE_BANDED,
                        CHEST_COMPRESSION_LOWER_RIGHT_SCORE_BANDED
                    ].map(Number)
                );
            }

            // Abdomen Compression

            const abdomenValues = [
                ABDOMEN_COMPRESSION_LEFT,
                ABDOMEN_COMPRESSION_RIGHT,
                ABDOMEN_COMPRESSION_LEFT_SCORE_BANDED,
                ABDOMEN_COMPRESSION_RIGHT_SCORE_BANDED
            ];

            if (abdomenValues.some(isInvalid)) {
                ABDOMEN_COMPRESSION_MAX_VALUE = "Missing";
                ABDOMEN_COMPRESSION_SCORE = "Missing";
            } else {
                ABDOMEN_COMPRESSION_MAX_VALUE = Math.max(
                    Number(ABDOMEN_COMPRESSION_LEFT),
                    Number(ABDOMEN_COMPRESSION_RIGHT)
                );

                ABDOMEN_COMPRESSION_SCORE = Math.min(
                    Number(ABDOMEN_COMPRESSION_LEFT_SCORE_BANDED),
                    Number(ABDOMEN_COMPRESSION_RIGHT_SCORE_BANDED)
                );
            }

            if (max_upp_rib_com == "Missing" || max_upp_rib_com == null) {
                var topc = 0;
            } else {
                var topc = sliding_scale(
                    max_upp_rib_com,
                    ABDOMEN_COMPRESSION_GOOD,
                    ABDOMEN_COMPRESSION_WEAK,
                    ABDOMEN_HI_SCORE,
                    0
                );
            }

            if (max_bottom_rib_com == "Missing" || max_bottom_rib_com == null) {
                var botc = 0;
            } else {
                var botc = sliding_scale(
                    max_bottom_rib_com,
                    ABDOMEN_COMPRESSION_GOOD,
                    ABDOMEN_COMPRESSION_WEAK,
                    ABDOMEN_HI_SCORE,
                    0
                );
            }

            new Variable(
                template,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_01_SCORE`,
                `Result from Automotive Assessments Workflow`,
                topc.toFixed(3),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_VALUE`,
                `Result from Automotive Assessments Workflow`,
                CHEST_COMPRESSION_MAX_VALUE.toString(),
                "General",
                false,
                true
            );

            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                CHEST_COMPRESSION_SCORE.toString(),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_VALUE`,
                `Result from Automotive Assessments Workflow`,
                ABDOMEN_COMPRESSION_MAX_VALUE.toString(),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_ABDOMEN_COMPRESSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                ABDOMEN_COMPRESSION_SCORE.toString(),
                "General",
                false,
                true
            );
            // Fallback any NaN values to Missing
            const safe = (x) => {
                const num = Number(x);
                return isNaN(num) ? "Missing" : num;
            };

            const ac = safe(ABDOMEN_COMPRESSION_SCORE);
            const cc = safe(CHEST_COMPRESSION_SCORE);
            const mod = safe(modifier_total);

            let CHEST_ABDOMEN_SCORE;
            let CHEST_ABDOMEN_SCORE_NOMOD;
            let CHEST_ABDOMEN_POINTS;

            if ([ac, cc, mod].some((val) => val === "Missing")) {
                CHEST_ABDOMEN_SCORE = "Missing";
                CHEST_ABDOMEN_SCORE_NOMOD = "Missing";
                CHEST_ABDOMEN_POINTS = "Missing";
            } else {
                // All values are valid numbers
                // @ts-ignore
                CHEST_ABDOMEN_SCORE_NOMOD = Math.min(ac, cc);
                CHEST_ABDOMEN_SCORE = Math.min(CHEST_ABDOMEN_SCORE_NOMOD + modifier_total, 100);
                CHEST_ABDOMEN_SCORE = Math.max(CHEST_ABDOMEN_SCORE, 0);
                CHEST_ABDOMEN_POINTS = (CHEST_ABDOMEN_SCORE / 100) * 0.625;
            }

            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_SCORE`,
                `Result from Automotive Assessments Workflow`,
                CHEST_ABDOMEN_SCORE.toString(),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_POINTS`,
                `Result from Automotive Assessments Workflow`,
                CHEST_ABDOMEN_POINTS.toString(),
                "General",
                false,
                true
            );
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `${m}_${occ}_CHEST_ABDOMEN_SCORE_NOMOD`,
                `Result from Automotive Assessments Workflow`,
                CHEST_ABDOMEN_SCORE_NOMOD.toString(),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_CHEST_MODIFIER_FRONT_PASSENGER_TOTAL`,
                `Result from Automated Assessments Workflow`,
                modifier_total.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_ABDOMEN_COMPRESSION_MAX_02_SCORE`,
                `Result from Automotive Assessments Workflow`,
                botc.toFixed(3),
                "General",
                false,
                true
            );

            let maxc = Math.max(max_upp_rib_com, max_bottom_rib_com);
            new Variable(
                template,
                `ABDOMEN_MAX_COMPRESSION`,
                `Result from Automotive Assessments Workflow`,
                maxc.toFixed(3),
                "General",
                false,
                true
            );

            let abdomen_score = Math.min(topc, botc);
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `ABDOMEN_MAX_COMPRESSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                abdomen_score.toFixed(3),
                "General",
                false,
                true
            );

            let chest_score = get_variable_value_and_ENCAP_band_score(
                status,
                `CHEST_MAX_COMPRESSION_SCORE`,
                "float",
                template
            );
            let chest_abdomen_score = Math.min(chest_score, abdomen_score);
            new_variable_and_new_euroNCAP_banded_score(
                template,
                `CHEST_ABDOMEN_FINAL_SCORE`,
                `Result from Automotive Assessments Workflow`,
                chest_abdomen_score.toFixed(3),
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_ABDOMEN_CAPPING_LIMIT`,
                `Result from Automotive Assessments Workflow`,
                maxc > ABDOMEN_COMPRESSION_CAPPING_LIMIT ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );
        }
    }
}

function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
