/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
knee_femur_and_pelvis_score();

/**
 * Performs the knee, femur and pelvis score calculation for the specified models
 */
function knee_femur_and_pelvis_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `knee, femur and pelvis`;
    let status;

    let occupants = ["DRIVER", "FRONT_PASSENGER"];
    let sides = ["LEFT", "RIGHT"];
    let side_scores = {}; // To be populated with score for each side
    for (let m of models) {
        for (let occ of occupants) {
            for (let side of sides) {
                LogPrint(`Calculating ${m} ${occ} ${side} ${body_region_label} score...`);

                /* Create a status object to track whether REPORTER Variables are all present and valid.
                 * <success> is initially true but will be set to false if anything missing or invalid. */
                status = { success: true, missing: [], invalid: [] };

                /* Knee, femur and pelvis score and modifiers */

                /* Driver */
                let acetabular_score;
                let femur_comp_exceedence_score;

                /* Passenger */
                let femur_compression_score;

                /* Shared */
                let knee_score;

                switch (occ) {
                    case "DRIVER":
                        acetabular_score = get_variable_value(
                            status,
                            `${m}_${occ}_ACETABULAR_FORCE_${side}_SCORE`,
                            "float"
                        );
                        femur_comp_exceedence_score = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_FEMUR_COMPRESSION_EXCEEDENCE_SCORE`,
                            "float"
                        );
                        knee_score = get_variable_value(status, `${m}_${occ}_${side}_KNEE_COMPRESSION_SCORE`, "float");

                        break;
                    case "FRONT_PASSENGER":
                        femur_compression_score = get_variable_value(
                            status,
                            `${m}_${occ}_${side}_FEMUR_COMPRESSION_SCORE`,
                            "float"
                        );
                        break;
                    default:
                        LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                        Exit();
                }
                /* Final scores all start at zero and will remain so if any variables were missing or invalid */
                side_scores[side] = 0;

                /* If we have all the required variables, calculate the final scores */
                if (status.success) {
                    /* Score calculation is different for different occupants */
                    switch (occ) {
                        case "DRIVER":
                            /* Left/right score is minimum of acetabular, femur and knee scores, summed with modifiers */
                            side_scores[side] = Math.max(
                                0,
                                Math.min(acetabular_score, femur_comp_exceedence_score, knee_score)
                            );
                            break;
                        case "FRONT_PASSENGER":
                            /* Left/right score is minimum of femur score, summed with modifiers */
                            side_scores[side] = femur_compression_score;

                            break;
                        default:
                            LogError(`Unexpected occupant type "${occ}" in ${body_region_label} calculation.`);
                            Exit();
                    }

                    LogPrint(`${m} ${occ} ${side} ${body_region_label} score = ${side_scores[side]}`);
                } else {
                    warn_about_missing_or_invalid_variables(
                        status,
                        `${m} ${occ} ${body_region_label} score calculation`
                    );
                }

                let side_score_var = new Variable(
                    templ,
                    `${m}_${occ}_KNEE_FEMUR_PELVIS_${side}_FINAL_SCORE`,
                    `Final ${side} ${body_region_label} score`,
                    side_scores[side].toString(),
                    "String",
                    false,
                    true
                );
            }

            let submarine_mod = get_variable_value(status, `${m}_${occ}_SUBMARINE_MODIFIER`, "float");
            let knee_airbag_deployment_mod = get_variable_value(
                status,
                `${m}_${occ}_KNEE_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`,
                "float"
            );

            /* Overall score for each occupant is the minimum of the left and right side scores, plus the submarine modifier and knee airbag deployment modifier */
            let knee_femur_pelvis_final_score =
                Math.min(side_scores.LEFT, side_scores.RIGHT) + submarine_mod + knee_airbag_deployment_mod;

            if (knee_femur_pelvis_final_score < 0.0) {
                knee_femur_pelvis_final_score = 0.0;
            }

            let knee_femur_pelvis_final_score_var = new Variable(
                templ,
                `${m}_${occ}_KNEE_FEMUR_PELVIS_FINAL_SCORE`,
                `Final ${body_region_label} score`,
                knee_femur_pelvis_final_score.toString(),
                "String",
                false,
                true
            );
        }
    }
}
