/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
head_score();

/**
 * Performs the head score calculation for the specified models
 * Note that we assume 2nd row female sits on Rear Driver Side. In future need to allow sitting on either side
 */
function head_score() {
    let templ = Template.GetCurrent();
    let models = get_model_list();
    let body_region_label = `head`;

    let occupants = ["DRIVER", "FRONT_PASSENGER", "REAR_DRIVER_SIDE"];

    for (let m of models) {
        /* Sum up all front row dummies modifiers for this body region, later,
         * the lower injury score between driver & front passenger minus the sum of modifiers,
         * will become the final front row dummies score for this body region */
        let head_frt_row_modifer = 0.0;
        /* Collect all front row modifiers */
        for (let occ of occupants) {
            if (occ == "REAR_DRIVER SIDE") {
                break;
            }
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} modifier...`);
            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };
            let haz_ab_mod;
            let inc_ab_mod;
            let ab_con_mod;
            let steering_column_status = { success: true, missing: [], invalid: [] };
            let steering_column_intrusion_z_score;
            /* Common for front row dummies */
            haz_ab_mod = get_variable_value(status, `${m}_${occ}_HEAD_HAZARDOUS_AIRBAG_DEPLOYMENT_MODIFIER`, "int");
            inc_ab_mod = get_variable_value(status, `${m}_${occ}_HEAD_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`, "int");
            ab_con_mod = get_variable_value(status, `${m}_${occ}_HEAD_AIRBAG_CONTACT_MODIFIER`, "int");
            /* Driver specific */
            if (occ == "DRIVER") {
                steering_column_intrusion_z_score = get_variable_value(
                    steering_column_status,
                    `${m}_STRUCTURE_STEERING_COLUMN_VERTICAL_INTRUSION_SCORE`,
                    "float"
                );
                if (!steering_column_status.success) {
                    Window.Message("Message", occ + " steering column missing, modifier set to zero.");
                    steering_column_intrusion_z_score = 0.0;
                }
            }
            /* If we have all the required variables, calculate the final modifiers */
            let head_mod = 0.0;
            if (status.success) {
                if (occ == "DRIVER") {
                    /* For the driver, also add the steering column intrusion modifier */
                    head_mod = haz_ab_mod + inc_ab_mod + ab_con_mod + steering_column_intrusion_z_score;
                    head_frt_row_modifer = head_frt_row_modifer + head_mod;
                }
                if (occ == "FRONT_PASSENGER") {
                    head_mod = haz_ab_mod + inc_ab_mod + ab_con_mod;
                    head_frt_row_modifer = head_frt_row_modifer + head_mod;
                }
            }
        }
        let head_frt_row_modifer_var = new Variable(
            templ,
            `${m}_TOTAL_HEAD_FRONT_ROW_MODIFIER`,
            `Total head front row modifier`,
            head_frt_row_modifer.toString(),
            "String",
            false,
            true
        );
        /* Finished sum of front row modifiers calculation, start scoring */

        /* Later we create a REPORTER variable to display an explanatory message if any capping
         * limit exceeded. Please note that for C-NCAP 2024 Front cases (FRB, MPDB),
         * the total score consists of: (1) front row score for each body region,
         * (2) 2nd row female score for each body region, (3) 2nd row child
         * To support both front row occupants + 2nd row female,
         * two separate capping limit messages are created as follows: */
        let capping_limit_explanation_frt_row = "";
        let capping_limit_explanation_sec_row_female = "";

        /* Use this var to compare and store the front row score and capping notice */
        let frt_row_score_head = 0.0;
        let frt_row_cap_head = "";
        let head_score_driver = 0.0;
        let head_score_front_passenger = 0.0;

        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Head scores and modifiers */
            let head_hic_score;
            let head_hic_capping_limit;
            let head_tms_exceedence_score;
            let head_tms_capping_limit;
            let haz_ab_mod;
            let inc_ab_mod;
            let ab_con_mod;
            let steering_column_intrusion_z_score;
            let steering_column_status = { success: true, missing: [], invalid: [] };
            let head_2nd_impact;
            let fwd_disp_mod;

            /* Shared variables among all 3 occupants */
            haz_ab_mod = get_variable_value(status, `${m}_${occ}_HEAD_HAZARDOUS_AIRBAG_DEPLOYMENT_MODIFIER`, "int");
            inc_ab_mod = get_variable_value(status, `${m}_${occ}_HEAD_INCORRECT_AIRBAG_DEPLOYMENT_MODIFIER`, "int");
            ab_con_mod = get_variable_value(status, `${m}_${occ}_HEAD_AIRBAG_CONTACT_MODIFIER`, "int");

            /* Front row variables */
            if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                /* Shared by front row */
                head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC_SCORE`, "float");
                head_hic_capping_limit = get_variable_value(status, `${m}_${occ}_HEAD_HIC_CAPPING_LIMIT`);
                head_tms_exceedence_score = get_variable_value(
                    status,
                    `${m}_${occ}_HEAD_THREE_MS_EXCEEDENCE_SCORE`,
                    "float"
                );
                head_tms_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_HEAD_THREE_MS_EXCEEDENCE_CAPPING_LIMIT`
                );
                /* Driver specific */
                if (occ == "DRIVER") {
                    steering_column_intrusion_z_score = get_variable_value(
                        steering_column_status,
                        `${m}_STRUCTURE_STEERING_COLUMN_VERTICAL_INTRUSION_SCORE`,
                        "float"
                    );
                    if (!steering_column_status.success) {
                        //Window.Message("Message", occ + " steering column missing, modifier set to zero.");
                        steering_column_intrusion_z_score = 0.0;
                    }
                }
            }
            /* 2nd row female variables */
            if (occ == "REAR_DRIVER_SIDE") {
                /* 2nd row female */
                head_hic_score = get_variable_value(status, `${m}_${occ}_HEAD_HIC_REAR_PASSENGER_SCORE`, "float");
                head_hic_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_HEAD_HIC_REAR_PASSENGER_CAPPING_LIMIT`
                );
                head_tms_exceedence_score = get_variable_value(
                    status,
                    `${m}_${occ}_HEAD_THREE_MS_EXCEEDENCE_REAR_PASSENGER_SCORE`,
                    "float"
                );
                head_tms_capping_limit = get_variable_value(
                    status,
                    `${m}_${occ}_HEAD_THREE_MS_EXCEEDENCE_REAR_PASSENGER_CAPPING_LIMIT`
                );
                head_2nd_impact = get_variable_value(status, `${m}_${occ}_HEAD_SECONDARY_IMPACT_MODIFIER`);
                /* Note that 2nd row female head forward displacement modifier can be -0.5 or -1.0 */
                fwd_disp_mod = get_variable_value(status, `${m}_${occ}_HEAD_FORWARD_DISPLACEMENT_MODIFIER`, "float");
            }

            /* Final scores all start at zero and will remain so if any variables were missing or invalid */
            let head_score = 0.0;

            /* Record the modifiers */
            let head_mod = 0.0;
            let head_score_no_mod = 0.0;

            /* Previously, capping limits start as exceeded and will remain so if any variables were missing or invalid
             * However, now we want them to start from blank, so if any dummies missing it will not show "*" */
            let head_capping_limit = "";

            /* If we have all the required variables, calculate the final scores */
            /* Ref. Table 3-12 on Chapter.III of C-NCAP 2024 Management Regulation */
            if (status.success) {
                /* Head score is sum of score with modifiers.
                 * For the driver, there is an extra steering column vertical displacement modifer 0~-1 points.
                 * Start with front row occupants
                 * Note that for front row occupants, the deduction value is sum of all front row regional modifiers
                 * This is because for C-NCAP 2024 FRB load case, only one final front row score is needed, and it is
                 * the "lower injury score between driver and front passenger" minus all regional "front row mods"*/
                if (occ == "DRIVER" || occ == "FRONT_PASSENGER") {
                    /* Driver / front passenger head score */
                    if (occ == "DRIVER") {
                        head_mod = haz_ab_mod + inc_ab_mod + ab_con_mod + steering_column_intrusion_z_score;
                        head_score_no_mod = Math.min(head_hic_score, head_tms_exceedence_score);
                        head_score_driver = Math.max(0, head_score_no_mod + head_mod);
                        head_score = head_score_driver;
                        LogPrint(`${m} ${occ} head score = ${head_score}`);
                    }
                    if (occ == "FRONT_PASSENGER") {
                        head_mod = haz_ab_mod + inc_ab_mod + ab_con_mod;
                        head_score_no_mod = Math.min(head_hic_score, head_tms_exceedence_score);
                        head_score_front_passenger = Math.max(0, head_score_no_mod + head_mod);
                        head_score = head_score_front_passenger;
                        LogPrint(`${m} ${occ} head score = ${head_score}`);
                    }

                    /* Check capping limits. If any individual limits are exceeded, an asterisk is
                     * displayed for all parent body regions. */
                    if (head_hic_capping_limit == "" && head_tms_capping_limit == "") {
                        head_capping_limit = "";
                    } else {
                        head_capping_limit = "*";
                        capping_limit_explanation_frt_row = `*Capping limit exceeded`;
                        /* If any of front row dummies capped, the front row cap is on with asterisk */
                        frt_row_cap_head = "*";
                    }
                }

                /* 2nd row female head score with modifiers
                 * Note that we assume 2nd row female sits on Rear Driver Side
                 * In future need to allow sitting on either side */
                if (occ == "REAR_DRIVER_SIDE") {
                    head_mod = haz_ab_mod + inc_ab_mod + ab_con_mod + fwd_disp_mod;
                    /* According to 1.2.1.1.2.1 in Chapter III of C-NCAP 2024,
                     * if no head secondary impact, only 3ms acc taken in scoring */
                    if (head_2nd_impact == "Yes") {
                        //LogPrint(`Head secondary impact yes`);
                        head_score = Math.min(head_hic_score, head_tms_exceedence_score) + head_mod;
                        head_score_no_mod = Math.min(head_hic_score, head_tms_exceedence_score);
                    }
                    if (head_2nd_impact == "No") {
                        //LogPrint(`Head secondary impact no`);
                        head_score = head_tms_exceedence_score + head_mod;
                        head_score_no_mod = head_tms_exceedence_score;
                    }
                    /* Bound the score between upper and lower limits */
                    head_score = Math.max(head_score, 0.0);
                    LogPrint(`${m} ${occ} head score = ${head_score}`);

                    /* Check capping limits. If any individual limits are exceeded, an asterisk is
                     * displayed for all parent body regions. */
                    if (head_hic_capping_limit == "" && head_tms_capping_limit == "") {
                        head_capping_limit = "";
                    } else {
                        capping_limit_explanation_sec_row_female = `*Capping limit exceeded`;
                    }
                }
            } else {
                warn_about_missing_or_invalid_variables(status, `${m} ${occ} ${body_region_label} score calculation`);
            }

            /* Overall scores */
            let head_score_var = new Variable(
                templ,
                `${m}_${occ}_HEAD_FINAL_SCORE`,
                "Final head score",
                head_score.toString(),
                "String",
                false,
                true
            );
            let head_mod_var = new Variable(
                templ,
                `${m}_${occ}_HEAD_MODIFIERS`,
                "Final head modifiers",
                head_mod.toString(),
                "String",
                false,
                true
            );
            let head_score_no_mod_var = new Variable(
                templ,
                `${m}_${occ}_HEAD_SCORE_NO_MOD`,
                "Head score before applying modifiers",
                head_score_no_mod.toString(),
                "String",
                false,
                true
            );
            /* Capping limits */
            let head_capping_limit_var = new Variable(
                templ,
                `${m}_${occ}_HEAD_CAPPING_LIMIT`,
                "Head capping limit (*asterisk if limit exceeded; empty string otherwise)",
                head_capping_limit,
                "String",
                false,
                true
            );
        }

        frt_row_score_head = Math.min(head_score_driver, head_score_front_passenger);

        let frt_row_score_head_var = new Variable(
            templ,
            `${m}_FRONT_ROW_HEAD_SCORE`,
            `Front row head final score`,
            frt_row_score_head.toString(),
            "String",
            false,
            true
        );
        let frt_row_cap_head_var = new Variable(
            templ,
            `${m}_FRONT_ROW_HEAD_CAPPING_LIMIT`,
            `Front row head capping limit asterisk`,
            frt_row_cap_head,
            "String",
            false,
            true
        );
        let capping_limit_explanation_frt_row_var = new Variable(
            templ,
            `${m}_FRONT_ROW_HEAD_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation - Head - Front row`,
            capping_limit_explanation_frt_row,
            "String",
            false,
            true
        );
        let capping_limit_explanation_sec_row_female_var = new Variable(
            templ,
            `${m}_SECOND_ROW_FEMALE_HEAD_CAPPING_LIMIT_EXPLANATION`,
            `Capping limit explanation - Head - 2nd row female`,
            capping_limit_explanation_sec_row_female,
            "String",
            false,
            true
        );
    }
}
