run_test();

/**
 * Cross-platform function for the Window.GetFile method. Instead of taking a string, it takes
 * in a list of extensions and constructs the appropriate filter string based on the OS. It also takes
 * the parameters for save and initial.
 * @param {string[]|string} [extensions=[]] - List of file extensions (e.g., [".csv", ".txt"]) or a single extension string (e.g., ".csv")
 * @param {boolean} [save=false] - Whether the dialog is for saving a file
 * @param {string} [initial=""] - Initial file path or directory name. If a file path is provided, the parent directory is used. If a directory is provided, it's used as-is.
 * @returns {string|null} - Selected filename as a string, or null if the user cancels
 */
function GetFile(extensions = [], save = false, initial = "") {
    /* Check that extensions is a string or an array. */
    if (typeof extensions === "string") {
        /* If the user passes a string of delimited values (with ; commas or spaces to separate the extensions) instead of an array,
         * split the string into an array using the appropriate delimiter. The regex splits on commas, semicolons, or spaces. */
        extensions = extensions.split(/[,; ]+/);
    } else if (!Array.isArray(extensions)) {
        LogError("Function GetFile: extensions parameter must be a string or an array of strings.");
        return null;
    }

    // Replace all the *'s at the start of the string (this is generally before the first dot of the extension).
    // We do this because we add the * later on using the joiner variable.
    extensions = extensions.map((ext) => ext.replace(/^\**/, ""));

    const is_linux = Unix();

    /**
     * The filter adds a * before each extension (excluding the first) and joins them with either spaces (Linux) or semicolons (Windows).
     */
    let file, filter;
    const joiner = is_linux ? " *" : "; *";
    const file_separator = is_linux ? "/" : "\\";

    // Adjust initial path for Linux and Windows if needed. On Linux, convert backslashes to forward slashes.
    // On Windows, convert forward slashes to backslashes. This ensures the initial path is in the correct format for the OS.
    if (is_linux && initial) {
        initial = initial.replace(/\\/g, file_separator);
    }

    if (!is_linux && initial) {
        initial = initial.replace(/\//g, file_separator);
    }

    filter = extensions.map((ext) => `${ext}`).join(joiner);

    if (initial && File.Exists(initial)) {
        // For Window.GetFile, if a directory is used but is not properly specified with a / or \\ at the end, then the appropriate
        // delimiter (e.g. forward slash) is added to force it to use the correct directory. Otherwise the function may misinterpret it as a file.
        if (File.IsDirectory(initial) && !initial.endsWith(file_separator)) {
            initial = initial + file_separator;
        }
        file = Window.GetFile(filter, save, initial);
    } else {
        file = Window.GetFile(filter, save);
    }
    return file;
}

function run_test() {
    let ans1, ans2;
    let test_file_model_path = "CANCEL";

    /** check if the template contains SimT items - in which case, ask the user if they can select the SimVT settings file*/
    let templ = Template.GetCurrent();
    let master_page = templ.GetMaster();
    let simvt_this_item = Item.GetFromName(master_page, `#SIMVT T/HIS check and do assessment`);

    if (simvt_this_item) {
        let ans = Window.Message(
            `Select SimVT settings file`,
            `Do you want to select a SimVT settings file?`,
            Window.YES | Window.NO
        );
        if (ans == Window.YES) {
            let simvt_settings_file = Window.GetFile(".simvt");
            if (simvt_settings_file) {
                new Variable(
                    templ,
                    "SIMVT_SETTINGS_FILE",
                    "SimVT settings file",
                    simvt_settings_file,
                    "File(absolute)",
                    false,
                    true
                );

                // if the user selected a test file then try and extract the test file from the simvt settings file

                let simvt_settings_json = ReadJSON(simvt_settings_file);
                let test_file_model_tag = simvt_settings_json.selected_channels_data[0].ref.model_tag;
                for (let mc_definition of simvt_settings_json.model_container_definitions) {
                    if (mc_definition.model_tag === test_file_model_tag) {
                        test_file_model_path = mc_definition.file_path;
                    }
                }

                if (test_file_model_path) {
                    ans1 = Window.Message(
                        `Select different test file?`,
                        `The test file defined in the SimVT settings file was:\n\n` +
                            `${test_file_model_path}\n\n` +
                            `Would you like to select a different test file?`,
                        Window.YES | Window.NO
                    );
                } else {
                    Window.Message(`No test file found in SimVT settings file`, `Please select a test file manually.`);
                }
            }
        } else if (ans == Window.NO) {
            ans2 = Window.Message(
                `Select test file`,
                `Select the test file (ISO-MME or CSV).`,
                Window.OK | Window.CANCEL
            );
        }
    } else {
        ans2 = Window.Message(`Select test file`, `Select the test file (ISO-MME or CSV).`, Window.OK | Window.CANCEL);
    }

    if (ans1 == Window.YES || ans2 == Window.OK) {
        test_file_model_path = GetFile(".csv *.iso *.mme *.chn *.mmi");
    }

    /** when we get here test_file_model_path will either be a valid path or "CANCEL" */
    new Variable(templ, "TEST_FILE", "Test file", test_file_model_path, "File(absolute)", false, true);
    // To support both names: in AA the 'TEST_FILE' is same as SimVT 'TEST_DATA_PATH' 3-Sep-2025
    new Variable(templ, "TEST_DATA_PATH", "Test file", test_file_model_path, "File(absolute)", false, true);
}
