/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
// Duke Xiao 14-Aug-2025

// This is a script run in REPORTER to read CSV files from OUTPUT_DIR and populate the table
// in the template. The 1st application is in C-NCAP 2024 Far Side VTC official format template
// where parts energies are extracted, and we need to write a parts info table.
// In the C-NCAP 2024 Far Side VTC official template,
// This function is run after getCncapPartsInfo() from
// "%TEMPLATE_DIR%/../../scripts/automotive_assessments/modules/pre/reporter_cncap_parts_info.mjs";
// And we should already have:
//  *  %OUTPUT_DIR%/CNCAP_Parts_Airbag.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Centre_Console.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Seatbelt.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Seat_Bucket.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Seat_Cushion.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Seat_Side_Member.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Seat_Front_Rear_Crossmembers.csv
//  *  %OUTPUT_DIR%/CNCAP_Parts_Seat_4Bar_Linkage.csv

reporter_table_from_csv();

function reporter_table_from_csv() {
    // If the template is C-NCAP 2024 Far Side VTC official summary ...
    let templ = Template.GetCurrent();
    if (templ.filename.includes(`CNCAP_2024_Far_Side`)) {
        let output_dir = Variable.GetFromName(templ, "OUTPUT_DIR");
        // Map the CSV in OUTPUT_DIR to the Table item name in REPORTER
        // Please check in REPORTER the tables are named same as table_name property
        let csv_list = [
            {
                path: "CNCAP_Parts_Airbag.csv",
                table_name: "Table_Airbag"
            },
            {
                path: "CNCAP_Parts_Centre_Console.csv",
                table_name: "Table_Centre_Console"
            },
            {
                path: `CNCAP_Parts_Seatbelt.csv`,
                table_name: "Table_Seatbelt"
            },
            {
                path: `CNCAP_Parts_Seat_Bucket.csv`,
                table_name: "Table_Seat_Bucket"
            },
            {
                path: `CNCAP_Parts_Seat_Cushion.csv`,
                table_name: "Table_Seat_Cushion"
            },
            {
                path: `CNCAP_Parts_Seat_Side_Member.csv`,
                table_name: "Table_Seat_Side_Member"
            },
            {
                path: `CNCAP_Parts_Seat_Front_Rear_Crossmembers.csv`,
                table_name: "Table_Seat_Front_Rear_Crossmembers"
            },
            {
                path: `CNCAP_Parts_Seat_4Bar_Linkage.csv`,
                table_name: "Table_Seat_4Bar_Linkage"
            }
        ];
        for (let csv of csv_list) {
            let csv_file_path = `${output_dir.value}/${csv.path}`;
            if (File.Exists(csv_file_path)) {
                LogPrint(`Start processing ${csv_file_path}`);
                cncap_2024_far_side_part_table(csv_file_path, csv.table_name);
            } else {
                Handle_missing_data(csv.table_name);
                LogWarning(
                    `Cannot find CSV: ${csv_file_path}, please check if you have defined Workflows Automotive Assessments data for this structure in C-NCAP 2024 Far Side Sled, in the 1st model that you have specified as input.`
                );
            }
        }
        Exit();
    }
}

function cncap_2024_far_side_part_table(csv_file_path, table_name) {
    let templ = Template.GetCurrent();
    let obj_table;
    for (let p = 0; p < templ.pages; p++) {
        obj_table = Item.GetFromName(templ.GetPage(p), table_name);
        if (obj_table != null && obj_table.type == Item.TABLE) {
            LogPrint(`Start populating table: ${table_name} on page ${p}`);
            let f = new File(csv_file_path, File.READ);
            let line = "";
            line = f.ReadLongLine(); // Skip 1st row
            let row = 2; // start from 2nd row, actual indice is row-1 since indice starts from zero
            // @ts-ignore In REPORTER JS API, f.ReadLongLine returns string or File.EOF so type mismatch expected
            while ((line = f.ReadLongLine()) != File.EOF) {
                if (row > obj_table.rows) {
                    // If running out of rows, add another row
                    // @ts-ignore
                    obj_table.InsertRow();
                }
                let arr_line = line.split(",");
                let cell_prop = {
                    fontSize: 8,
                    textColour: Colour.Black(),
                    fontName: "Microsoft YaHei UI",
                    justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
                };
                if (csv_file_path.includes("Airbag")) {
                    cell_prop.text = arr_line[0];
                    obj_table.SetCellProperties(cell_prop, row - 1, 0);
                    cell_prop.text = arr_line[1];
                    obj_table.SetCellProperties(cell_prop, row - 1, 1);
                    row++;
                } else {
                    cell_prop.text = arr_line[0];
                    obj_table.SetCellProperties(cell_prop, row - 1, 0);
                    cell_prop.text = arr_line[1];
                    obj_table.SetCellProperties(cell_prop, row - 1, 1);
                    cell_prop.text = arr_line[2];
                    obj_table.SetCellProperties(cell_prop, row - 1, 2);
                    cell_prop.text = arr_line[3];
                    obj_table.SetCellProperties(cell_prop, row - 1, 3);
                    cell_prop.text = arr_line[4];
                    obj_table.SetCellProperties(cell_prop, row - 1, 4);
                    cell_prop.text = arr_line[5];
                    obj_table.SetCellProperties(cell_prop, row - 1, 5);
                    row++;
                }
            }
            f.Close();
            LogPrint(`Finished populating table: ${table_name} on page ${p}`);
            break;
        }
    }
}

/* function to handle the missing data, set the table with "Missing data" */
function Handle_missing_data(table_name) {
    let templ = Template.GetCurrent();
    let obj_table;
    for (let p = 0; p < templ.pages; p++) {
        obj_table = Item.GetFromName(templ.GetPage(p), table_name);
        if (obj_table != null && obj_table.type == Item.TABLE) {
            obj_table.InsertRow(1);
            let cell_prop = {
                fontSize: 8,
                text: `Missing Data`,
                textColour: Colour.Black(),
                fontName: "Microsoft YaHei UI",
                justify: Reporter.JUSTIFY_CENTRE | Reporter.JUSTIFY_MIDDLE
            };
            obj_table.SetCellProperties(cell_prop, 1, 0);
        }
    }
}
