/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
SubjectiveModifier_InitialiseAllModifiers();

/* ES6 classes can't be redeclared within the same REPORTER JavaScript realm so use ES5-style
 * "class" syntax using function instead. */

/**
 * Constructor for SubjectiveModifier object
 * @param {string} name REPORTER Variable name for this subjective modifier
 * @param {string|Object} description Description; appears in table when user sets modifiers
 * @param {?string} occupant Occupant string
 * @param {any[]} values List of allowable values for this subjective modifier
 */
function SubjectiveModifier(name, description, occupant, values) {
    this.name = name;
    /** description could be a string or an object with "English" and "Chinese" language options
     * if it is a string then use it as is. If it is an object then use the English value by default
     * unless the template name ends in "_CN" in which case use the Chinese value */
    if (typeof description === "object") {
        let templ = Template.GetCurrent();
        let language = "English";
        /** if the template filename contains "_CN.or" then set the language to Chinese */
        if (/_CN\.or/i.test(templ.filename)) {
            language = "Chinese";
        }

        this.description = description[language];
    } else {
        this.description = description;
    }

    this.occupant = occupant;
    this.values = values;
    // NEW LOGIC: Support both old [0, -20] format and new { yesValue: -20, noValue: 0 }
    // if (Array.isArray(values)) {
    //     this.values = values;
    //     this.yesValue = values[1];
    //     this.noValue = values[0];
    // } else if (typeof values === "object" && values !== null) {
    //     this.values = [values.noValue, values.yesValue]; // for compatibility
    //     this.yesValue = values.yesValue;
    //     this.noValue = values.noValue;
    // } else {
    //     throw new Error("Invalid 'values' format for SubjectiveModifier: must be array or yes/no object");
}

/**
 * Gets the subjective modifiers from the modifiers.json file for the specified regulation,
 * crash test and version.
 * @param {string} regulation The regulation e.g. "EuroNCAP"
 * @param {string} crash_test The crash test e.g. "MPDB"
 * @param {string} version The version e.g. "2020"
 * @param {?string} [occupant=null] Occupant string (if applicable)
 * @returns {SubjectiveModifier[]}
 * @example
 * let modifiers = SubjectiveModifier_GetModifiers("EuroNCAP", "MPDB", "2020");
 */
function SubjectiveModifier_GetModifiers(regulation, crash_test, version, occupant = null) {
    let templ = Template.GetCurrent();
    let template_dir = templ.GetVariableValue("TEMPLATE_DIR");
    if (template_dir == null) {
        throw new Error(`Unable to get value of TEMPLATE_DIR in SubjectiveModifier.GetModifiers`);
    }
    let subj_mod_dir = `${template_dir}/../../scripts/automotive_assessments/subjective_modifiers`;
    let subj_mod_fname = `${subj_mod_dir}/${regulation}/${crash_test}/${version}/modifiers.json`;
    if (!File.Exists(subj_mod_fname)) {
        throw new Error(`Could not find modifiers file for ${regulation} ${crash_test} ${version}: ${subj_mod_fname}`);
    }
    LogPrint(`Reading subjective modifiers file: ${subj_mod_fname}`);
    let f = new File(subj_mod_fname, File.READ);
    /* REPORTER JS API lacks f.ReadAll() so read file line by line */
    let line;
    let text = "";
    // @ts-ignore In REPORTER JS API, f.ReadLongLine returns string or File.EOF so type mismatch expected
    while ((line = f.ReadLongLine()) != File.EOF) {
        text = text + line;
    }
    f.Close();
    let json = JSON.parse(text);

    let modifiers = [];
    if (Array.isArray(json.modifiers) && json.modifiers.length > 0) {
        for (let mod of json.modifiers) {
            /* Ignore if occupant is specified and doesn't match */
            if (occupant != null && mod.occupant != occupant) {
                continue;
            }

            modifiers.push(new SubjectiveModifier(mod.name, mod.description, mod.occupant, mod.values));
        }
    } else {
        throw new Error(`Could not find list of subjective modifiers in file. Check JSON syntax: ${subj_mod_fname}`);
    }
    return modifiers;
}

/**
 * Creates and initialises all the subjective modifiers defined in the
 * modifiers.json file for the current template
 */
function SubjectiveModifier_InitialiseAllModifiers() {
    let templ = Template.GetCurrent();

    /* We will search for Set Modifiers items with names beginning with an identifiable string */
    let set_mod_str = `set_modifiers`;

    /* Search for relevant items in all pages */
    let pages = templ.GetAllPages();
    for (let p = 0; p < pages.length; p++) {
        let items = pages[p].GetAllItems();
        for (let item of items) {
            if (item.name.substring(0, set_mod_str.length) == set_mod_str) {
                /* Expect item name to be of the form:
                 *
                 *     `set_modifiers~${regulation}~${crash_test}~${version}`
                 *
                 * or if there are occupant specific modifiers:
                 *
                 *     `set_modifiers~${regulation}~${crash_test}~${version}~{occupant}`
                 *
                 * Extract the relevant fields */
                let fields = item.name.substring(set_mod_str.length + 1).split("~");
                if (fields.length < 3) {
                    LogError(
                        `Could not extract Regulation, Crash Test, Version and Occupant from item name` +
                            `"${item.name} on page ${p + 1}. Skipping subjective modifiers initialisation.`
                    );
                    continue;
                }
                let regulation = fields[0];
                let crash_test = fields[1];
                let version = fields[2];
                let models = ["M1"]; // TODO add support for multiple models

                /* Get the occupant if it exists */
                let occupant = null;

                if (fields.length > 3) {
                    occupant = fields[3];
                }

                SubjectiveModifier_InitialiseModifiers(regulation, crash_test, version, models, occupant);
            }
        }
    }
}

/**
 * Sets the head protection rating on the press of a button
 * @param {string} variable_name the name of the variable which has to be set
 * @param {string} rating could be either "Good", "Marginal", "Acceptable", "Poor"
 * @example
 * update_head_protection_rating("M1_DRIVER_HEAD_PROTECTION_RATING", "ACCEPTABLE");
 */
function update_head_protection_rating(variable_name, rating) {
    var templ = Template.GetCurrent();
    let new_variable = new Variable(templ, variable_name, `Updated the variable`, rating, "String", false, true);
    recalculate_results_IIHS();
}

function recalculate_results_IIHS() {
    /*This part of the code must be completed to redraw the driver plot and overall score*/
    //have to re-draw driver and passenger plots
    generate_item(`#AAW REPORTER scoring`);
}

/**
 * Both structural and overall ratings are updated to poor based on fuel leak or high voltage system integrity fail
 * @param {string} result the result, OK or Poor
 */
function change_fuel_leak(result) {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    for (let m of models) {
        if (result == "Poor") {
            new Variable(
                templ,
                `${m}_FUEL_VOLTAGE_DOWNGRADE_DEMERIT`,
                `If a significant fuel leak or compromise of the high-voltage system occurs, both the structural and overall ratings may be downgraded to poor`,
                result,
                "string",
                false,
                true
            );
        } else if (result == "OK") {
            new Variable(
                templ,
                `${m}_FUEL_VOLTAGE_DOWNGRADE_DEMERIT`,
                `If a significant fuel leak or compromise of the high-voltage system occurs, both the structural and overall ratings may be downgraded to poor`,
                result,
                "string",
                false,
                true
            );
        }
    }

    recalculate_results_IIHS();
}

/**
 * Increases or decreases the structural downgrade variable on the press of a button
 * @param {number} increment the amount of which the variable must be changed
 * @example
 * change_structural_downgrade_IIHS(1);
 */
function change_structural_downgrade_IIHS(increment) {
    let templ = Template.GetCurrent();
    let models = get_model_list();

    let structural_downgrade = 0;
    let downgrade_levels = 0;
    for (let m of models) {
        //During the first time ever this function is called. It enters the if block. Initializes the STRUCTURAL DOWNGRADE value to 0.
        LogPrint(`Updating ${m} structural downgrade...`);
        /* Create a status object to track whether REPORTER Variables are all present and valid.
         * <success> is initially true but will be set to false if anything missing or invalid. */
        let status = { success: true, missing: [], invalid: [] };

        structural_downgrade = get_variable_value(status, `${m}_STRUCTURAL_DOWNGRADE`, "int");

        downgrade_levels = structural_downgrade + increment;

        if (downgrade_levels >= 0 && downgrade_levels <= 3) {
            // downgrading can only be 0, 1, 2 or 3 levels
            let updated_structural_downgrade = new Variable(
                templ,
                `${m}_STRUCTURAL_DOWNGRADE`,
                `A variable to keep track of how much downgrade to be applied to structural rating`,
                downgrade_levels.toString(),
                "string",
                false,
                true
            );
        }
    }

    recalculate_results_IIHS();
}

/**
 * Initialises the default values of the subjective modifiers for the specified regulation,
 * crash test and version, for each of the specified models
 * @param {string} regulation The regulation e.g. "EuroNCAP"
 * @param {string} crash_test The crash test e.g. "MPDB"
 * @param {string} version The version e.g. "2020"
 * @param {string[]} models Array of model numbers to be initialised
 * @param {string} occupant Occupant string
 * @example
 * SubjectiveModifier_InitialiseModifiers("EuroNCAP", "MPDB", "2020", [1, 2]);
 */
function SubjectiveModifier_InitialiseModifiers(regulation, crash_test, version, models, occupant) {
    try {
        let templ = Template.GetCurrent();
        let modifiers = SubjectiveModifier_GetModifiers(regulation, crash_test, version, occupant);

        let suffix = get_variable_suffix(regulation);

        for (let mod of modifiers) {
            for (let m of models) {
                /* Constructor will overwrite existing variable. */
                let mod_var = new Variable(
                    templ,
                    `${m}_${mod.name}_${suffix}`,
                    get_unicode_text_with_padding(`${mod.description} [${mod.values.join(", ")}]`),
                    mod.values[0],
                    `General`,
                    false,
                    true
                );
            }
        }
    } catch (e) {
        LogError(`${e}.\n${e.stack}`);
    }
}

/**
 * Prompts the user to set the values of the subjective modifiers for the specified regulation,
 * crash test and version, for each of the specified models
 * @param {string} regulation The regulation e.g. "EuroNCAP"
 * @param {string} crash_test The crash test e.g. "MPDB"
 * @param {string} version The version e.g. "2020"
 * @param {string} occupant The occupant
 * @param {string[]} models Array of model numbers for which modifier values are to be set
 * @example
 * SubjectiveModifier_UserSetModifiers("EuroNCAP", "MPDB", "2020", ["M1", "M2"]);
 */
function SubjectiveModifier_UserSetModifiers(regulation, crash_test, version, models, occupant) {
    let templ = Template.GetCurrent();
    let modifiers = SubjectiveModifier_GetModifiers(regulation, crash_test, version, occupant);

    let suffix = get_variable_suffix(regulation);

    /* Construct a list of Variables to include a copy of each modifier for each model */
    let variables = [];
    for (let modifier of modifiers) {
        for (let m of models) {
            variables.push(`${m}_${modifier.name}_${suffix}`);
        }
    }

    replace_language_constant_variable_strings_with_value(variables);

    LogPrint(`Setting subjective modifiers...`);
    let edit = templ.EditVariables(
        "Edit Subjective Modifiers",
        "Double click to edit the values:",
        false,
        variables,
        Variable.DESCRIPTION,
        false
    );

    replace_language_constants();

    /* If user clicked OK, update variable values if user entered valid values */

    if (edit) {
        let invalid_entries = [];
        /* Loop through all the edited variables */
        for (let name in edit) {
            /* For each one, we search for a matching modifier */
            let name_match = false;
            let name_regex;
            for (let mod of modifiers) {
                name_regex = new RegExp(`[MT]\\d+_${mod.name}_${suffix}`);

                if (name.match(name_regex)) {
                    /* For the matching modifier, we check whether the user-entered value is one of
                     * those listed in the modifier definition */
                    let valid_value = null;
                    for (let value of mod.values) {
                        /* First, we check whether numerical values match */
                        let num = Number.parseFloat(edit[name]);
                        if (!Number.isNaN(num) && num == value) valid_value = num;
                        /* We also check whether string values match (either is acceptable) */ else if (
                            edit[name] == value
                        )
                            valid_value = value;
                        /* Change the Variable value if it's valid - constructor will overwrite existing variables */
                        if (valid_value != null) {
                            let mod_var = new Variable(
                                templ,
                                name,
                                get_unicode_text_with_padding(`${mod.description} [${mod.values.join(", ")}]`),
                                valid_value,
                                `General`,
                                false,
                                true
                            );
                            break;
                        }
                    }
                    /* If the user-entered value didn't match any of the modifier's list of valid
                     * values, add it to the list of invalid entries */
                    if (valid_value == null) {
                        invalid_entries.push({ modifier: mod, user_value: edit[name] });
                    }
                    name_match = true;
                    break;
                }
            }
            if (!name_match) {
                LogError(`Failed to find modifier to match edited variable: ${name}`);
            }
        }

        /* Now we have edited subjective modifiers, we need to recalculate the scores. */
        generate_item(`#AAW REPORTER scoring`);

        /* We also need to regenerate any dummy_image items to reflect the updated scores. First,
         * we scan for dummy_image items. We will search for items with names beginning with an
         * identifiable string. */
        let dummy_image_item_str = `dummy_image`;
        let pages = templ.GetAllPages();
        for (let p = 0; p < pages.length; p++) {
            let items = pages[p].GetAllItems();
            for (let item of items) {
                if (item.name.substring(0, dummy_image_item_str.length) == dummy_image_item_str) {
                    /* Report any items erroneously given the identifier name */
                    if (item.type != Item.IMAGE_FILE) {
                        LogError(`Item "${item.name}" on page ${p + 1} is not an Image File item.`);
                        continue;
                    }
                    /* Generate the dummy_image item */
                    generate_item(item.name);
                }
            }
        }

        if (invalid_entries.length > 0) {
            let msg = `Scores have been recalculated for valid entries but there were invalid entries for some of the modifiers:`;
            for (let entry of invalid_entries) {
                msg =
                    msg +
                    `\n${entry.modifier.description} [${entry.modifier.values.join(", ")}]: ${
                        entry.user_value
                    } is invalid`;
            }
            LogWarning(msg);
            Window.Warning(`Set Modifiers`, msg);
        } else {
            let msg = `Scores have been recalculated.`;
            LogPrint(msg);
            Window.Information(`Set Modifiers`, msg);
        }
    } else {
        LogPrint(`Set Modifiers cancelled by user.`);
    }
}

/**
 * Returns the suffix for the variable name based on the regulation
 * @param {string} regulation Regulation e.g. "EuroNCAP"
 * @returns {string}
 */
function get_variable_suffix(regulation) {
    /* IIHS regulation uses DEMERIT suffix, others use MODIFIER */
    return regulation.toUpperCase() == "IIHS" ? "DEMERIT" : "MODIFIER";
}
