/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */

do_automotive_assessment_scoring();

/**
 * Generates any scoring scripts that may be present in the template. Body region script items are
 * located on individual body region pages so that they can be added/removed in a modular way.
 * Rather than just letting them generate when their page is reached, we need to generate them all
 * first, in a certain order, so that the body region scores are available for the overall score
 * and "draw images" scripts, which typically appear earlier in the template.
 */
function do_automotive_assessment_scoring() {
    let templ = Template.GetCurrent();
    let models = ["M1"]; // For now, just run for M1 only

    /* Create the model list variable so that the score functions can run */
    let model_list_var = new Variable(
        templ,
        `MODEL_LIST`,
        `List of models processed in template (e.g. "M1, M2, M3")`,
        models.join(", "),
        "String",
        false,
        true
    );
    /* Generate any scoring scripts that may be present in the order given here */
    let scoring_scripts = [
        `head`,
        `head_1`,
        `neck`,
        `head_and_neck`,
        `chest`,
        `chest_1`,
        `abdomen`,
        `abdomen_1`,
        `chest_and_abdomen`,
        `shoulder`,
        `thigh_and_hip`,
        `pelvis`,
        `upper_leg`,
        `knee_femur_and_pelvis`,
        `femur_and_knee`,
        `tibia`,
        `lower_leg`,
        `lower_leg_foot_and_ankle`,
        `structural_rating`,
        `restraint_rating`,
        `overall`,
        `overall_1`,
        `rating_dial`,
        `draw_images`,
        `structural_graphs`,
        `summary`
    ];
    for (let script of scoring_scripts) {
        generate_item(`#AAW Scoring Script ${script}`, false);
    }

    /**
     * When re-running scoring (e.g., executing the subjective modifier),
     * the variable value need to be "Pass" or "Fail".
     * Therefore, we need to replace the variable value again.
     */
    replace_language_constants();
}

/**
 * Returns the appropriate fill colour for the specified score
 * @param {number} score The body region score
 * @returns {string}
 */
function get_fill_colour_EuroNCAP(score) {
    /* Euro NCAP colours */
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const ORANGE = "#ff9700";
    const BROWN = "#753f2a";
    const RED = "#e00700";
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    if (score < 0.01) {
        return RED;
    } else if (score < 1.33) {
        return BROWN;
    } else if (score < 2.67) {
        return ORANGE;
    } else if (score < 4.0) {
        return YELLOW;
    } else {
        return GREEN;
    }
}

/**
 * Returns the appropriate fill colour for the specified score
 * @param {number} score The body region score
 * @param {number} [max] The maximum score for the body region
 * @returns {string}
 */
function get_fill_colour_CNCAP(score, max = 4) {
    /* Euro NCAP colours */
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const RED = "#e00700";
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    if (score < 0.01) {
        return RED;
    } else if (score < max) {
        return YELLOW;
    } else {
        return GREEN;
    }
}

/**
 * Returns the appropriate fill colour for the specified score.
 * Accepts numbers or strings (including "Missing").
 * @param {number|string} score The body region score
 * @returns {string}
 */
function get_fill_colour_EuroNCAP_2026(score) {
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const ORANGE = "#ff9700";
    const BROWN = "#753f2a";
    const RED = "#e00700";
    const WHITE = "#ffffff";
    const MISSING = "#f3f7fb";

    if (score == null || score === "") {
        return WHITE;
    }

    // Convert strings to numbers, and check if valid
    const numericScore = Number(score);
    if (isNaN(numericScore)) {
        return MISSING; // fallback for badly formatted strings
    }

    if (numericScore < 0.01) {
        return RED;
    } else if (numericScore < 33.33) {
        return BROWN;
    } else if (numericScore < 66.66) {
        return ORANGE;
    } else if (numericScore < 99.99) {
        return YELLOW;
    } else {
        return GREEN;
    }
}

/**
 * Returns the appropriate fill colour for the specified score
 * @param {number} score The body region score
 * @returns {string}
 */
function get_fill_colour_KNCAP(score) {
    /* KNCAP colours */
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const ORANGE = "#ff9700";
    const BROWN = "#753f2a";
    const RED = "#e00700";
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    if (score < 0.01) {
        return RED;
    } else if (score < 1.33) {
        return BROWN;
    } else if (score < 2.67) {
        return ORANGE;
    } else if (score < 4.0) {
        return YELLOW;
    } else {
        return GREEN;
    }
}

/**
 * Returns the appropriate fill colour for the specified score
 * @param {number} score The body region score
 * @returns {string}
 */
function get_fill_colour_GNCAP(score) {
    /*  GNCAP colours */
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const ORANGE = "#ff9700";
    const BROWN = "#753f2a";
    const RED = "#e00700";
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    if (score < 0.01) {
        return RED;
    } else if (score < 1.33) {
        return BROWN;
    } else if (score < 2.67) {
        return ORANGE;
    } else if (score < 4.0) {
        return YELLOW;
    } else {
        return GREEN;
    }
}

function get_fill_colour_JNCAP(score) {
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const ORANGE = "#ff9700";
    const BROWN = "#753f2a";
    const RED = "#e00700";
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    if (score < 0.01) {
        return RED;
    } else if (score < 1.33) {
        return BROWN;
    } else if (score < 2.67) {
        return ORANGE;
    } else if (score < 4.0) {
        return YELLOW;
    } else {
        return GREEN;
    }
}

/**
 * Returns the appropriate fill colour for the specified score
 * @param {string} score Output of UN assessment is passed which is either a pass or a fail
 * @returns {string}
 */
function get_fill_colour_UN(score) {
    const GREEN = "#269b29";
    const RED = "#e00700";
    const WHITE = "#ffffff";
    if (score == null) {
        return WHITE;
    }
    if (score == "Pass") {
        return GREEN;
    }
    if (score == "Fail") {
        return RED;
    }
}

/**
 * Returns the appropriate fill colour for the specified rating
 * @param {string} score The body region rating
 * @returns {string}
 */
function get_fill_colour_IIHS(score) {
    /* IIHS colours */
    const GREEN = "#52B442"; // Green
    const YELLOW = "#FCE702"; // Yellow
    const ORANGE = "#F79800"; // Orange
    const RED = "#F40000"; // Red
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    if (score == "Poor") {
        return RED;
    } else if (score == "Marginal") {
        return ORANGE;
    } else if (score == "Acceptable") {
        return YELLOW;
    } else if (score == "Good") {
        return GREEN;
    } else {
        return RED;
    }
}

function get_fill_colour_USNCAP(score) {
    /* This is the scaling factor used in USNCAP templates which is used to divide the net probability to obtain relative risk */
    const USNCAP_SCALING_FACTOR = 0.15;

    /* These are threshold values used in USNCAP templates to define the rating bar */
    const THRESHOLD_USNCAP_1 = 0.67;
    const THRESHOLD_USNCAP_2 = 1;
    const THRESHOLD_USNCAP_3 = 1.33;
    const THRESHOLD_USNCAP_4 = 2.67;

    /* USNCAP colours */
    const GREEN = "#269b29";
    const YELLOW = "#ffcc00";
    const ORANGE = "#ff9700";
    const BROWN = "#753f2a";
    const RED = "#e00700";
    /* White (for uncoloured regions) */
    const WHITE = "#ffffff";

    if (score == null) {
        return WHITE;
    }

    /* Relative risk is the */
    score = score / USNCAP_SCALING_FACTOR;

    if (score < THRESHOLD_USNCAP_1) {
        return GREEN;
    } else if (score < THRESHOLD_USNCAP_2) {
        return YELLOW;
    } else if (score < THRESHOLD_USNCAP_3) {
        return ORANGE;
    } else if (score < THRESHOLD_USNCAP_4) {
        return BROWN;
    } else {
        return RED;
    }
}

/**
 * This function checks if the directory is valid, where the output draw images as stored and retrieved by REPORTER
 * @param {String}  output_dir output_dir is a parameter here which refers to a string value pointing to the output dir which is check if valid
 * @returns {Boolean} Returns the boolean value TRUE: if dir is valid;FALSE: if dir is invalid
 *
 */

function is_valid_output_dir(output_dir) {
    if (!output_dir) {
        LogError(`Could not find OUTPUT_DIR variable for writing occupant image files.`);
        return false;
    } else if (!File.IsDirectory(output_dir)) {
        LogError(`OUTPUT_DIR is not a valid directory: ${output_dir}`);
        return false;
    } else {
        return true;
    }
}

/**
 * Returns the worst rating from the given ratings
 * @param {string[]} ratings The array of ratings
 */
function return_worst_IIHS_rating(ratings) {
    let rating_order = ["Poor", "Marginal", "Acceptable", "Good"];
    let worst_rating = "Poor";

    for (let rating of rating_order) {
        if (ratings.includes(rating)) {
            worst_rating = rating;
            break;
        }
    }

    return worst_rating;
}

/**
 * Returns an object with good, marginal, acceptable and poor properties, each containing
 * the number of occurrences of the corresponding rating in the given array of ratings
 * @param {string[]} ratings Array of ratings, e.g. ["Good", "Good", "Poor", "Marginal"]
 * @returns {object}
 * @example
 * let rating_counts = count_IIHS_ratings(["Good", "Good", "Poor", "Marginal"]);
 * // rating_counts = { good: 2, acceptable: 0, marginal: 1, poor: 1 }
 */
function count_IIHS_ratings(ratings) {
    let rating_counts = {
        good: 0,
        acceptable: 0,
        marginal: 0,
        poor: 0
    };

    for (let rating of ratings) {
        if (rating == "Good") {
            rating_counts.good++;
        } else if (rating == "Acceptable") {
            rating_counts.acceptable++;
        } else if (rating == "Marginal") {
            rating_counts.marginal++;
        } else if (rating == "Poor") {
            rating_counts.poor++;
        } else {
            LogError(`Invalid rating in <count_IIHS_ratings>: ${rating}`);
        }

        rating_counts[rating]++;
    }

    return rating_counts;
}

/**
 * Returns the overall IIHS rating from the counts of each rating
 * @param {object} ratings_count Object containing the counts of each rating
 * @returns {string}
 * @example
 * let overall_rating = get_overall_IIHS_rating_from_count({ good: 2, acceptable: 0, marginal: 1, poor: 1 });
 */
function get_overall_IIHS_rating_from_count(ratings_count) {
    /* Rating is the one that there is most of.
     * If there is a tie, then take the lower rating. */

    let rating = "Not Computed";

    if (
        ratings_count.poor > 0 &&
        ratings_count.poor >= ratings_count.marginal &&
        ratings_count.poor >= ratings_count.acceptable &&
        ratings_count.poor >= ratings_count.good
    ) {
        rating = "Poor";
    } else if (
        ratings_count.marginal > 0 &&
        ratings_count.marginal >= ratings_count.acceptable &&
        ratings_count.marginal >= ratings_count.good
    ) {
        rating = "Marginal";
    } else if (ratings_count.acceptable > 0 && ratings_count.acceptable >= ratings_count.good) {
        rating = "Acceptable";
    } else {
        rating = "Good";
    }

    return rating;
}

/**
 * The overall structural rating cannot be more than one rating level better than the worst rating
 * for each individual strucutral rating.
 *
 * This function returns an object with a boolean to indicate if the rating passed to this function
 * should be downgraded or not, the worst rating out of the individual ratings and the rating to
 * downgrade to.
 * @param {string} rating Rating to check
 * @param {object} ratings_count Object containing the counts of each rating
 * @returns {object}
 * @example
 * let clamp = clamp_IIHS_structural_rating("Good", { good: 2, acceptable: 0, marginal: 1, poor: 1 });
 * // clamp = { downgraded: true, worst_rating: "Poor", clamped_rating: "Marginal" }
 */
function clamp_IIHS_strucural_rating(rating, ratings_count) {
    let downgraded = false;
    let downgrade_to = rating;
    let worst_rating = "";

    if (rating == "Good" && ratings_count.poor > 0) {
        downgrade_to = "Marginal";
        downgraded = true;
        worst_rating = "Poor";
    }

    if (rating == "Good" && ratings_count.marginal > 0) {
        downgrade_to = "Acceptable";
        downgraded = true;
        worst_rating = "Marginal";
    }

    if (rating == "Acceptable" && ratings_count.poor > 0) {
        downgrade_to = "Marginal";
        downgraded = true;
        worst_rating = "Poor";
    }

    return { downgraded: downgraded, worst_rating: worst_rating, downgrade_to: downgrade_to };
}

/**
 * Downgrades the rating based on the number of levels to downgrade
 * @param {string} rating Rating to downgrade
 * @param {number} downgrade_level Number of levels to downgrade
 * @returns {string}
 * @example
 * let downgraded_rating = downgraded_IIHS_rating("Good", 2);  // downgraded_rating = "Marginal"
 */
function downgrade_IIHS_rating(rating, downgrade_level) {
    let downgraded_rating = rating;

    if (rating == "Good") {
        if (downgrade_level == 1) {
            downgraded_rating = "Acceptable";
        } else if (downgrade_level == 2) {
            downgraded_rating = "Marginal";
        } else if (downgrade_level >= 3) {
            downgraded_rating = "Poor";
        }
    } else if (rating == "Acceptable") {
        if (downgrade_level == 1) {
            downgraded_rating = "Marginal";
        } else if (downgrade_level >= 2) {
            downgraded_rating = "Poor";
        }
    } else if (rating == "Marginal") {
        if (downgrade_level >= 1) {
            downgraded_rating = "Poor";
        }
    }

    return downgraded_rating;
}

/**
 * Returns the pass/fail from an array of assessments
 * @param {string[]} assessments The array of assessments
 * @return {string}
 */
function return_pass_or_fail(assessments) {
    const passString = "Pass";
    const failString = "Fail";

    if (assessments == ["failString"]) {
        return failString; /* Final scores is set to fail and will remain so if any variables were missing or invalid */
    } else if (assessments == ["passString"]) {
        return passString; /* To provide Nij the "Pass" string */
    }

    let allPass = true;

    let ratingResult = failString;
    for (let assessment of assessments) {
        if (assessment == failString || assessment != passString) {
            allPass = false;
            break;
        }
    }

    ratingResult = allPass ? passString : failString;

    return ratingResult;
}
/**
 * Returns the overall capping limit, given the capping limits for individual assessments. Used for EuroNCAP, GNCAP and KNAP protocols
 * If all the individual capping limits are blank strings "", then an blank string is returned.
 * If any individual capping limit is not a blank string, then a "*" is returned.
 * @param {string[]} assessments The array of assessment capping limits
 * @return {string}
 */

function return_capping_limit(assessments) {
    let capping_fail = "*";
    let capping_pass = "";
    for (let assessment of assessments) {
        if (assessment != "") {
            return capping_fail;
        }
    }
    return capping_pass;
}

/**
 * This function calculates the final joint probability.
 * The calculation for the final probability is :
 * joint probability = 1 - (1 - socre1) * (1 - score2) * (1 - score3);
 * @param {number[]}  input_scores an array is given as arguments when the function is called. It contains all the individual probabilities that we need.
 * @returns {Number} Returns the final joint probability.
 * The product_scores = (1 - socre1) * (1 - score2) * (1 - score3);
 */
function calculate_joint_probability_USNCAP(input_scores) {
    let product_scores = 1;
    for (let input of input_scores) {
        product_scores = product_scores * (1 - input);
    }
    let final_prob = 1 - product_scores;
    return final_prob;
}

/**
 * This function calculates the relative risk.
 * The calculation for relative risk is :
 * relative risk = joint_probabiltiy / 0.15;
 * @param {Number}  joint_probability The joint probabililty.
 * @returns {Number} Returns the relative risk of the occupant.
 */
function calculate_relative_risk_USNCAP(joint_probability) {
    /* This is the scaling factor used in USNCAP templates which is used to divide the net probability to obtain relative risk */
    const USNCAP_SCALING_FACTOR = 0.15;

    let relative_risk = joint_probability / USNCAP_SCALING_FACTOR;
    return relative_risk;
}

/* 
 This function creates rating bar for all USNCAP templates by taking in parameters suct as output directory, model and occupant details from draw_images.js script files.  
*/
function draw_ratings_bar_USNCAP(output_dir, model, occupant) {
    /* These are threshold values used in USNCAP templates to define the rating bar */
    const THRESHOLD_USNCAP_1 = 0.67;
    const THRESHOLD_USNCAP_2 = 1;
    const THRESHOLD_USNCAP_3 = 1.33;
    const THRESHOLD_USNCAP_4 = 2.67;

    var GREEN_COL = "#3b9e32"; // Green
    var YELLOW_COL = "#ffeb00"; // Yellow
    var ORANGE_COL = "#f49700"; // Orange
    var BROWN_COL = "#753f2a"; // Brown
    var RED_COL = "#d40022"; // Red
    var BLACK_COL = "#000000"; // Black

    /* Create a status object to track whether REPORTER Variables are all present and valid.
     * <success> is initially true but will be set to false if anything missing or invalid. */
    let status = { success: true, missing: [], invalid: [] };

    let score = get_variable_value(status, `${model}_${occupant}_RELATIVE_RISK`, "float");

    var img = new Image(640, 65);

    img.lineWidth = 1;
    img.fontSize = 10;
    img.font = "Helvetica";

    var diff = 5;

    // Draw the bar

    img.lineColour = GREEN_COL;
    img.fillColour = GREEN_COL;
    img.Polygon([10, 25, 134, 25, 134, 45, 10, 45]);
    img.Text(6, 60, "0");

    img.lineColour = BLACK_COL;
    img.fillColour = BLACK_COL;
    img.Star(72, 35, 8);
    img.Star(87 + diff, 35, 8);
    img.Star(57 - diff, 35, 8);
    img.Star(102 + 2 * diff, 35, 8);
    img.Star(42 - 2 * diff, 35, 8);

    img.lineColour = YELLOW_COL;
    img.fillColour = YELLOW_COL;
    img.Polygon([134, 25, 258, 25, 258, 45, 134, 45]);
    img.Text(130, 60, THRESHOLD_USNCAP_1.toString());
    img.lineColour = BLACK_COL;
    img.fillColour = BLACK_COL;
    img.Star(124 + 72 - 7.5 - diff, 35, 8);
    img.Star(124 + 72 + 7.5 + diff, 35, 8);
    img.Star(124 + 62 - 12.5 - 2 * diff, 35, 8);
    img.Star(124 + 82 + 12.5 + 2 * diff, 35, 8);

    img.lineColour = ORANGE_COL;
    img.fillColour = ORANGE_COL;
    img.Polygon([258, 25, 382, 25, 382, 45, 258, 45]);
    img.Text(254, 60, THRESHOLD_USNCAP_2.toString());
    img.lineColour = BLACK_COL;
    img.fillColour = BLACK_COL;
    img.Star(248 + 72, 35, 8);
    img.Star(248 + 87 + diff, 35, 8);
    img.Star(248 + 57 - diff, 35, 8);

    img.lineColour = BROWN_COL;
    img.fillColour = BROWN_COL;
    img.Polygon([382, 25, 506, 25, 506, 45, 382, 45]);
    img.Text(378, 60, THRESHOLD_USNCAP_3.toString());
    img.lineColour = BLACK_COL;
    img.fillColour = BLACK_COL;
    img.Star(372 + 72 - 7.5 - diff, 35, 8);
    img.Star(372 + 72 + 7.5 + diff, 35, 8);

    img.lineColour = RED_COL;
    img.fillColour = RED_COL;
    img.Polygon([506, 25, 630, 25, 630, 45, 506, 45]);
    img.Text(502, 60, THRESHOLD_USNCAP_4.toString());
    img.lineColour = BLACK_COL;
    img.fillColour = BLACK_COL;
    img.Star(496 + 72, 35, 8);

    // Draw the arrow to indicate where the rating is on the bar

    if (score <= THRESHOLD_USNCAP_1) var x = 10 + ((134 - 10) / THRESHOLD_USNCAP_1) * score;
    else if (score <= THRESHOLD_USNCAP_2)
        var x = 134 + ((258 - 134) * (score - THRESHOLD_USNCAP_1)) / (THRESHOLD_USNCAP_2 - THRESHOLD_USNCAP_1);
    else if (score <= THRESHOLD_USNCAP_3)
        var x = 258 + ((382 - 258) * (score - THRESHOLD_USNCAP_2)) / (THRESHOLD_USNCAP_3 - THRESHOLD_USNCAP_2);
    else if (score <= THRESHOLD_USNCAP_4)
        var x = 382 + ((506 - 382) * (score - THRESHOLD_USNCAP_3)) / (THRESHOLD_USNCAP_4 - THRESHOLD_USNCAP_3);
    else var x = 506 + ((630 - 506) * (score - THRESHOLD_USNCAP_4)) / (12 - THRESHOLD_USNCAP_4);

    if (x > 630) x = 630;

    img.lineColour = "black";
    img.fillColour = "black";
    img.Polygon([x, 25, x - 10, 5, x + 10, 5]);

    // Write out image

    img.Save(`${output_dir}/${model}_${occupant.toLowerCase()}_rating_plot.png`, Image.PNG);
}

/**
 * Returns the evaluation level (from 1-5) from the test score
 * according to Rating Table 1 in the JNCAP Protocol
 * @param {number} score Overall test score
 */
function get_JNCAP_evaluation_level(score) {
    const LEVEL1 = 6;
    const LEVEL2 = 7.5;
    const LEVEL3 = 9.0;
    const LEVEL4 = 10.5;

    if (score < LEVEL1) {
        return 1;
    } else if (score < LEVEL2) {
        return 2;
    } else if (score < LEVEL3) {
        return 3;
    } else if (score < LEVEL4) {
        return 4;
    } else {
        return 5;
    }
}

/**
 * Draws the evaluation level image for JNCAP
 * @param {number} level Evaluation level (1-5)
 * @returns {Image}
 */
function draw_JNCAP_evaluation_level_image(level) {
    let img = new Image(300, 300);

    let passed_colour = "#006869";
    let failed_colour = "#b2b3b2";

    img.lineWidth = 1;

    /* Level 1 */
    img.lineColour = passed_colour;
    img.fillColour = passed_colour;
    img.Rectangle(5, 300, 55, 255);

    /* Level 2 */
    if (level >= 2) {
        img.lineColour = passed_colour;
        img.fillColour = passed_colour;
    } else {
        img.lineColour = failed_colour;
        img.fillColour = failed_colour;
    }
    img.Rectangle(65, 300, 115, 230);

    /* Level 3 */
    if (level >= 3) {
        img.lineColour = passed_colour;
        img.fillColour = passed_colour;
    } else {
        img.lineColour = failed_colour;
        img.fillColour = failed_colour;
    }
    img.Rectangle(125, 300, 175, 205);

    /* Level 4 */
    if (level >= 4) {
        img.lineColour = passed_colour;
        img.fillColour = passed_colour;
    } else {
        img.lineColour = failed_colour;
        img.fillColour = failed_colour;
    }
    img.Rectangle(185, 300, 235, 180);

    /* Level 5 */
    if (level >= 5) {
        img.lineColour = passed_colour;
        img.fillColour = passed_colour;
    } else {
        img.lineColour = failed_colour;
        img.fillColour = failed_colour;
    }
    img.Rectangle(245, 300, 295, 155);

    return img;
}

/**
 * Draws the head
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_head(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        94, 0, 102, 0, 112, 8, 116, 20, 116, 32, 114, 46, 106, 54, 102, 58, 94, 60, 86, 58, 82, 54, 74, 46, 72, 32, 72,
        20, 76, 8, 86, 0
    ]);
}

/**
 * Draws the head for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_head_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        75, 30, 82, 30, 90, 36, 93, 46, 93, 56, 91, 67, 85, 73, 82, 76, 75, 78, 69, 76, 66, 73, 59, 67, 58, 56, 58, 46,
        61, 36, 69, 30
    ]);
}

/**
 * Draws the neck
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_neck(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        94, 60, 102, 58, 106, 54, 112, 50, 112, 64, 106, 68, 102, 70, 94, 72, 86, 70, 82, 68, 76, 64, 76, 50, 82, 54,
        86, 58
    ]);
}

/**
 * Draws the neck for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_neck_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        75, 78, 82, 76, 85, 73, 90, 70, 90, 81, 85, 84, 82, 86, 75, 88, 69, 86, 66, 84, 61, 81, 61, 70, 66, 73, 69, 76
    ]);
}

/**
 * Draws the chest
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_chest(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        94, 72, 102, 70, 106, 68, 112, 64, 132, 64, 144, 66, 144, 72, 140, 78, 136, 82, 136, 136, 126, 138, 94, 138, 62,
        138, 52, 136, 52, 82, 48, 78, 44, 72, 44, 66, 56, 64, 76, 64, 82, 68, 86, 70
    ]);
}

/**
 * Draws the chest for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_chest_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        75, 88, 82, 86, 85, 84, 90, 81, 106, 81, 115, 83, 115, 88, 112, 92, 109, 96, 109, 139, 101, 140, 75, 140, 50,
        140, 42, 139, 42, 96, 38, 92, 35, 88, 35, 83, 45, 81, 61, 81, 66, 84, 69, 86
    ]);
}

/**
 * Draws the abdomen
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_abdomen(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([94, 138, 126, 138, 124, 174, 104, 176, 94, 176, 84, 176, 64, 174, 62, 138]);
}

/**
 * Draws the abdomen for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_abdomen_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([75, 140, 101, 140, 99, 169, 83, 171, 75, 171, 67, 171, 51, 169, 50, 140]);
}

/**
 * Draws the left arm, with a different position for the driver (to hold steering wheel)
 * @param {Image} img Image to draw on
 * @param {string} occ Occupant (DRIVER or a passenger)
 */
function draw_front_view_left_arm(img, occ) {
    img.fillColour = "#ffffff";
    if (occ == "DRIVER") {
        img.Polygon([144, 72, 150, 78, 164, 124, 166, 140, 154, 124, 144, 116, 136, 112, 136, 82, 140, 78]);
        img.Polygon([
            166, 140, 164, 146, 158, 152, 150, 152, 144, 146, 142, 134, 132, 132, 126, 126, 126, 120, 132, 116, 144,
            116, 154, 124
        ]);
    } else {
        img.Polygon([144, 72, 150, 78, 170, 136, 162, 130, 150, 136, 146, 142, 136, 120, 136, 82, 140, 78]);
        img.Polygon([
            170, 136, 170, 160, 168, 180, 158, 182, 146, 196, 138, 202, 132, 202, 128, 194, 128, 182, 142, 170, 156,
            168, 148, 150, 146, 142, 150, 136, 162, 130
        ]);
    }
}
/**
 * Draws the left arm for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Occupant (DRIVER or a passenger)
 */
function draw_front_view_left_arm_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([115, 88, 120, 92, 131, 129, 133, 142, 123, 129, 115, 123, 109, 120, 109, 96, 112, 92]);
}


/**
 * Draws the right arm, with a different position for the driver (to hold steering wheel)
 * @param {Image} img Image to draw on
 * @param {string} occ Occupant (DRIVER or a passenger)
 */
function draw_front_view_right_arm(img, occ) {
    img.fillColour = "#ffffff"; // White

    if (occ == "DRIVER") {
        img.Polygon([44, 72, 38, 78, 24, 124, 22, 140, 34, 124, 44, 116, 52, 112, 52, 82, 48, 78]);
        img.Polygon([
            22, 140, 24, 146, 30, 152, 38, 152, 44, 146, 46, 134, 56, 132, 62, 126, 62, 120, 56, 116, 44, 116, 34, 124
        ]);
    } else {
        img.Polygon([44, 72, 38, 78, 18, 136, 26, 130, 38, 136, 42, 142, 52, 120, 52, 82, 48, 78]);
        img.Polygon([
            18, 136, 18, 160, 20, 180, 30, 182, 42, 196, 50, 202, 56, 202, 60, 194, 60, 182, 46, 170, 32, 168, 40, 150,
            42, 142, 38, 136, 26, 130
        ]);
    }
}

/**
 * Draws the right arm for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Occupant (DRIVER or a passenger)
 */
function draw_front_view_right_arm_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([35, 88, 30, 92, 19, 129, 18, 142, 27, 129, 35, 123, 42, 120, 42, 96, 38, 92]);
}

/**
 * Draws the left femur
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_left_femur(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        94, 176, 104, 176, 124, 174, 170, 190, 174, 198, 174, 208, 170, 210, 162, 210, 154, 218, 142, 228, 116, 224, 94,
        218
    ]);
}

/**
 * Draws the left femur for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_left_femur_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        75, 171, 83, 171, 99, 169, 136, 182, 139, 188, 139, 196, 136, 198, 130, 198, 123, 204, 114, 212, 93, 209, 75,
        204
    ]);
}

/**
 * Draws the right femur
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_right_femur(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        94, 176, 84, 176, 62, 174, 18, 190, 14, 198, 14, 208, 18, 210, 26, 210, 34, 218, 46, 228, 72, 224, 94, 218
    ]);
}

/**
 * Draws the right femur for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_right_femur_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        75, 171, 67, 171, 50, 169, 14, 182, 11, 188, 11, 196, 14, 198, 21, 198, 27, 204, 37, 212, 58, 209, 75, 204
    ]);
}

/**
 * Draws the left tibia
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_left_tibia(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        144, 228, 144, 250, 148, 268, 152, 288, 158, 290, 168, 286, 172, 260, 176, 240, 176, 216, 172, 208, 162, 210,
        154, 218
    ]);
}

/**
 * Draws the left tibia for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_left_tibia_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        115, 212, 115, 230, 118, 244, 122, 260, 126, 262, 134, 259, 138, 238, 141, 222, 141, 203, 138, 196, 130, 198,
        123, 204
    ]);
}

/**
 * Draws the right tibia
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_right_tibia(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        44, 228, 44, 250, 40, 268, 36, 288, 30, 290, 20, 286, 16, 260, 12, 240, 12, 216, 16, 208, 26, 210, 34, 218
    ]);
}

/**
 * Draws the right tibia for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_right_tibia_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        35, 212, 35, 230, 32, 244, 29, 260, 24, 262, 16, 259, 13, 238, 10, 222, 10, 203, 13, 196, 21, 198, 27, 204
    ]);
}

/**
 * Draws the left foot
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_left_foot(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        150, 288, 144, 310, 150, 314, 152, 318, 154, 332, 160, 338, 178, 338, 186, 332, 188, 324, 180, 314, 172, 298,
        168, 286, 158, 290, 152, 288
    ]);
}

/**
 * Draws the left foot for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_left_foot_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        120, 260, 115, 278, 120, 281, 122, 284, 123, 296, 128, 300, 142, 300, 149, 296, 150, 289, 144, 281, 138, 268,
        134, 259, 126, 262, 122, 260
    ]);
}

/**
 * Draws the right foot
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_right_foot(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        38, 288, 44, 310, 38, 314, 36, 318, 34, 332, 28, 338, 10, 338, 2, 332, 0, 324, 8, 314, 16, 298, 20, 286, 30,
        290, 36, 288
    ]);
}

/**
 * Draws the right foot for child
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_front_view_right_foot_child(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        30, 260, 35, 278, 30, 281, 29, 284, 27, 296, 22, 300, 8, 300, 2, 296, 0, 289, 6, 281, 13, 268, 16, 259, 24, 262,
        29, 260
    ]);
}

/**
 * Draws the driver steering wheel
 * @param {Image} img Image to draw on
 */
function draw_front_view_steering_wheel(img) {
    img.lineColour = "#000000"; // Black
    img.lineWidth = 10;
    var oldLineCapStyle = img.lineCapStyle;
    img.lineCapStyle = Reporter.CAP_ROUND;
    img.Polyline([
        94, 84, 112, 86, 124, 94, 134, 102, 140, 112, 140, 122, 134, 132, 124, 140, 112, 148, 94, 150, 76, 148, 64, 140,
        54, 132, 48, 122, 48, 112, 54, 102, 64, 94, 76, 86, 94, 84
    ]);
    img.lineCapStyle = oldLineCapStyle;
}

/**
 * Draws the seatbelt
 * @param {Image} img Image to draw on
 * @param {string} occ Occupant (DRIVER or FRONT_PASSENGER)
 */
function draw_front_view_seatbelt(img, occ) {
    img.lineColour = "#000000"; // Black
    img.lineWidth = 14;

    switch (occ) {
        case "DRIVER":
            img.Polyline([154, 50, 52, 162]);
            img.Polyline([58, 166, 132, 166]);
            break;
        case "FRONT_PASSENGER":
            img.Polyline([34, 50, 136, 162]);
            img.Polyline([58, 166, 132, 166]);
            break;
        case "REAR_DRIVER_SIDE":
            img.Polyline([154, 50, 52, 162]);
            img.Polyline([130, 166, 56, 166]);
            break;
        case "REAR_PASSENGER_SIDE":
            img.Polyline([27, 70, 109, 160]);
            img.Polyline([46, 163, 106, 163]);
            break;
        default:
            LogError(`Unexpected occupant type "${occ}" in draw_seatbelt function.`);
            Exit();
            break;
    }
}

/**
 * Draws the head
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_side_view_head(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        26, 0, 38, 0, 42, 2, 50, 2, 52, 4, 56, 8, 62, 14, 64, 20, 66, 28, 64, 32, 64, 38, 68, 42, 68, 46, 62, 48, 60,
        50, 60, 64, 58, 66, 44, 64, 38, 64, 26, 56, 6, 54, 2, 44, 0, 34, 0, 20, 6, 12, 20, 2, 26, 2
    ]);
}

/**
 * Draws the head from left side
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_left_side_view_head(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        274, 0, 262, 0, 258, 2, 250, 2, 248, 4, 244, 8, 238, 14, 236, 20, 234, 28, 236, 32, 236, 38, 232, 42, 232, 46,
        238, 48, 240, 50, 240, 64, 242, 66, 256, 64, 262, 64, 274, 56, 294, 54, 298, 44, 300, 34, 300, 20, 294, 12, 280,
        2, 274, 2
    ]);
}

/**
 * Draws the neck
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_side_view_neck(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        6, 54, 24, 54, 28, 58, 38, 64, 42, 64, 46, 66, 46, 72, 44, 74, 44, 78, 34, 76, 18, 76, 12, 78, 8, 80, 10, 70, 8,
        64, 8, 58
    ]);
}

/**
 * Draws the neck from left side
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_left_side_view_neck(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        294, 54, 276, 54, 272, 58, 262, 64, 258, 64, 254, 66, 254, 72, 256, 74, 256, 78, 266, 76, 282, 76, 288, 78, 292,
        80, 290, 70, 292, 64, 292, 58
    ]);
}

/**
 * Draws the chest
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_side_view_chest(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        4, 94, 6, 90, 18, 80, 24, 78, 38, 80, 24, 78, 38, 78, 50, 82, 56, 84, 62, 90, 68, 98, 74, 108, 78, 122, 80, 132,
        82, 140, 82, 144, 78, 144, 76, 140, 68, 142, 48, 144, 30, 148, 20, 148, 14, 138, 10, 126, 4, 112, 4, 90
    ]);
}

/**
 * Draws the chest from left side
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_left_side_view_chest(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        296, 90, 296, 112, 290, 126, 286, 138, 280, 148, 270, 148, 252, 144, 232, 142, 224, 140, 222, 144, 218, 144,
        218, 140, 220, 132, 222, 122, 226, 108, 232, 98, 238, 90, 244, 84, 250, 82, 262, 78, 262, 80, 276, 78, 282, 80,
        294, 90, 296, 94
    ]);
}

/**
 * Draws the abdomen
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_side_view_abdomen(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        16, 150, 20, 148, 30, 148, 34, 146, 46, 146, 48, 144, 58, 142, 72, 140, 76, 140, 78, 148, 80, 156, 80, 164, 82,
        168, 76, 170, 58, 172, 52, 174, 26, 174, 22, 162, 18, 154
    ]);
}

/**
 * Draws the abdomen from left side
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_left_side_view_abdomen(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        284, 150, 280, 148, 270, 148, 266, 146, 254, 146, 252, 144, 242, 142, 228, 140, 224, 140, 222, 148, 220, 156,
        220, 164, 218, 168, 224, 170, 252, 172, 248, 174, 274, 174, 278, 162, 282, 154
    ]);
}

/**
 * Draws the right arm of driver
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon colour
 */
function draw_side_view_right_arm(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        4, 86, 12, 78, 20, 76, 30, 76, 40, 80, 48, 84, 64, 100, 90, 124, 96, 128, 104, 126, 114, 124, 124, 122, 134,
        120, 144, 118, 156, 114, 172, 98, 176, 96, 178, 96, 180, 94, 184, 94, 184, 98, 184, 100, 184, 102, 180, 106,
        186, 100, 188, 98, 184, 108, 178, 124, 168, 128, 144, 138, 118, 150, 98, 160, 88, 160, 84, 158, 74, 152, 54,
        138, 36, 126, 8, 102, 6, 92
    ]);
    img.Polygon([204, 92, 204, 104, 202, 110, 198, 116, 192, 122, 190, 120, 200, 98, 202, 92]);
}

/**
 * Draws the left arm of driver
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon colour
 */
function draw_left_side_view_left_arm(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        296, 86, 288, 78, 280, 76, 270, 76, 260, 80, 252, 84, 236, 100, 210, 124, 204, 128, 196, 126, 186, 124, 176,
        122, 166, 120, 156, 118, 144, 114, 128, 98, 124, 96, 122, 96, 120, 94, 116, 94, 116, 98, 116, 100, 116, 102,
        120, 106, 114, 100, 112, 98, 116, 108, 122, 124, 132, 128, 156, 138, 182, 150, 202, 160, 212, 160, 216, 158,
        226, 152, 246, 138, 264, 126, 292, 102, 294, 92
    ]);
    img.Polygon([96, 92, 96, 104, 98, 110, 102, 116, 108, 122, 110, 120, 100, 98, 98, 92]);
}

/**
 * Draws the right arm of any other occupant other than the driver
 * @param {Image} img Image to draw on
 */
function draw_side_view_right_arm_occupant(img) {
    img.fillColour = "#ffffff";
    img.Polygon([
        4, 86, 12, 78, 20, 76, 30, 76, 40, 80, 48, 84, 64, 100, 90, 124, 96, 128, 98, 128, 111, 127, 126, 126, 143, 127,
        160, 125, 168, 124, 176, 116, 185, 107, 191, 105, 197, 104, 194, 118, 190, 133, 181, 138, 172, 141, 163, 144,
        151, 147, 141, 150, 128, 153, 117, 156, 109, 157, 96, 161, 98, 160, 88, 160, 84, 158, 74, 152, 54, 138, 36, 126,
        8, 102, 6, 92
    ]);
}

/**
 * Draws the left arm of any other occupant other than the driver
 * @param {Image} img Image to draw on
 */
function draw_left_side_view_left_arm_occupant(img) {
    img.fillColour = "#ffffff";
    img.Polygon([
        296, 86, 288, 78, 280, 76, 270, 76, 260, 80, 252, 84, 236, 100, 210, 124, 204, 128, 202, 128, 189, 127, 174,
        126, 157, 127, 140, 125, 132, 124, 124, 116, 115, 107, 109, 105, 103, 104, 106, 118, 110, 133, 119, 138, 128,
        141, 137, 144, 149, 147, 159, 150, 172, 153, 183, 156, 191, 157, 204, 161, 202, 160, 212, 160, 216, 158, 226,
        152, 246, 138, 264, 126, 292, 102, 294, 92
    ]);
}

/**
 * Draws the right leg of driver
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon colour
 */
function draw_side_view_right_leg(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        86, 166, 106, 166, 116, 164, 144, 164, 154, 162, 164, 162, 170, 166, 176, 180, 186, 202, 200, 234, 212, 264,
        218, 270, 232, 272, 238, 274, 244, 278, 260, 278, 264, 280, 262, 290, 254, 292, 248, 294, 240, 292, 228, 292,
        222, 296, 212, 302, 206, 302, 202, 298, 202, 284, 194, 272, 174, 244, 164, 230, 156, 216, 154, 208, 154, 200,
        142, 206, 134, 208, 128, 206, 120, 208, 104, 210, 94, 212, 92, 182, 88, 160
    ]);
}

/**
 * Draws the left leg
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon colour
 */
function draw_left_side_view_left_leg(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        214, 166, 194, 166, 184, 164, 156, 164, 146, 162, 136, 162, 130, 166, 124, 180, 114, 202, 100, 234, 88, 264, 82,
        270, 68, 272, 62, 274, 56, 278, 40, 278, 36, 280, 38, 290, 46, 292, 52, 294, 60, 292, 72, 292, 78, 296, 88, 302,
        94, 302, 98, 298, 98, 284, 106, 272, 126, 244, 136, 230, 144, 216, 146, 208, 146, 200, 158, 206, 166, 208, 172,
        206, 180, 208, 196, 210, 206, 212, 208, 182, 212, 160
    ]);
}

/**
 * Draws the pelvis
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_side_view_pelvis(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        28, 174, 58, 174, 64, 172, 78, 170, 84, 166, 90, 176, 92, 186, 92, 212, 70, 214, 44, 218, 32, 206, 28, 196, 28,
        184
    ]);
}

/**
 * Draws the pelvis from left side
 * @param {Image} img Image to draw on
 * @param {string} fill_colour Polygon fill colour
 */
function draw_left_side_view_pelvis(img, fill_colour) {
    img.fillColour = fill_colour;
    img.Polygon([
        272, 174, 242, 174, 236, 172, 222, 170, 216, 166, 210, 176, 208, 186, 208, 212, 230, 214, 256, 218, 268, 206,
        272, 196, 272, 184
    ]);
}

/**
 * Draws the driver steering wheel
 * @param {Image} img Image to draw on
 * @param {string} steering_colour Polygon fill colour
 * */
function draw_side_view_steering_wheel(img, steering_colour) {
    img.lineColour = "#000000"; // Black
    img.lineWidth = 1;
    img.fillColour = steering_colour;
    img.Polygon([
        210, 60, 210, 70, 200, 98, 188, 126, 180, 142, 174, 150, 172, 150, 172, 144, 180, 118, 198, 74, 208, 60
    ]);
}

/**
 * Draws the driver steering wheel from left side
 * @param {Image} img Image to draw on
 * @param {string} steering_colour Polygon fill colour
 * */
function draw_left_side_view_steering_wheel(img, steering_colour) {
    img.lineColour = "#000000"; // Black
    img.lineWidth = 1;
    img.fillColour = steering_colour;
    img.Polygon([90, 60, 90, 70, 100, 98, 112, 126, 120, 142, 126, 150, 126, 150, 126, 144, 120, 118, 102, 74, 92, 60]);
}

/**
 * Converts a desired pixel height of upper case I to a font point size
 * @param {number} desired_pixel_height_upper_case_I Desired height of upper case I in pixels
 * @returns {number}
 */
function pixels_to_font_size(desired_pixel_height_upper_case_I) {
    var average_pixels_per_pt = determine_appropriate_font_size(); //this is based on screen res

    var required_font_point = Math.round(desired_pixel_height_upper_case_I / average_pixels_per_pt);

    LogPrint(
        "For your monitor font size " +
            required_font_point +
            "pt should generate text of height of " +
            desired_pixel_height_upper_case_I +
            "px"
    );

    return required_font_point;
}

function determine_appropriate_font_size() {
    var t = Template.GetCurrent();

    //var img_dir = t.GetVariableValue("IMAGES_DIR") + "/text_test/"; //used in testing

    var width = 1;
    var height = 200;
    var test_img = new Image(width, height);
    test_img.font = "Segoe UI";
    test_img.fontColour = "black";
    test_img.fontJustify = Reporter.JUSTIFY_CENTRE;

    var average_pixels_per_pt = 0;
    var samples = 0;
    for (var i = 6; i <= 100; i += 6) {
        test_img.fontSize = i;
        test_img.Text(width, height, "I");
        var nblack = width * height - test_img.PixelCount("white");

        //LogPrint("Font size " + i + "pt generates an 'I' of height " + nblack + " which is " + nblack/i + "pixels per pt");
        //test_img.Save(img_dir + "I_" + i+ "pt_npixels_"+nblack + ".png", Image.PNG);

        //clean with white rectangle
        test_img.lineColour = "white";
        test_img.fillColour = "white";
        test_img.lineWidth = 1;
        test_img.lineStyle = Reporter.LINE_SOLID;
        test_img.Rectangle(0, 0, width, height);

        average_pixels_per_pt += nblack / i;
        samples++;
    }

    average_pixels_per_pt = average_pixels_per_pt / samples;

    return average_pixels_per_pt;
}
