collate_report_contents();

/**
 * Collates items and creates a list of report contents for processing.
 */
function collate_report_contents() {
    let templ = Template.GetCurrent();

    /* This is set to true when at least one automotive assessment item is found.
     * If no images are found the generation is terminated (i.e. job control == `Abort`)
     * and a warning message pops up.
     */
    let found_automotive_assessment_item = false;

    /* Abort this item if JOB_CONTROL is set to `Abort` */
    let job_control = templ.GetVariableValue(`JOB_CONTROL`);
    if (job_control != null && job_control == `Abort`) {
        LogPrint(`Generation aborted (job control).`);
        Exit();
    }

    /* The report_contents.lst will be written to the output directory */
    let output_dir = get_expanded_variable_value(templ, `OUTPUT_DIR`);
    if (output_dir == null) {
        LogError(`Variable %OUTPUT_DIR% does not exist.`);
        return;
    }

    /* Create the output directory if it does not already exist */
    if (!File.IsDirectory(output_dir)) {
        let success = File.Mkdir(output_dir);
        if (!success) {
            LogError(`Unable to create output directory: ${output_dir}`);
            return;
        }
    }

    /* Create report_contents.lst, which will contain a list of all of the
     * automotive assessment contents that will need to be processed. */
    let f_contents_name = `${output_dir}/report_contents.lst`;
    let f_contents = new File(f_contents_name, File.WRITE);

    /* We will search for Image File items with names beginning with an identifiable string */
    let image_item_str = `automotive_assessment_image`;

    /* Loop through all pages in the template, searching for relevant items */
    const pages = templ.GetAllPages();
    for (let p = 0; p < pages.length; p++) {
        const items = pages[p].GetAllItems();
        for (const item of items) {
            if (item.name.substring(0, image_item_str.length) == image_item_str) {
                const pageNumber = p + 1;
                if (item.type == Item.D3PLOT) {
                    LogPrint(
                        `Found D3Plot ${item.filetype} item "${item.name}" on page ${pageNumber} with filename: ${item.file}`
                    );
                } else if (item.type != Item.IMAGE_FILE) {
                    /* Report any items erroneously given the identifier name */
                    LogError(`Item "${item.name}" on page ${pageNumber} is not an Image File item.`);
                    continue;
                } else {
                    /* Add the image filenames to the report_contents.lst. These
                     * filenames will be parsed in PRIMER to enable the required
                     * automotive assessments to be performed. */
                    LogPrint(`Found image "${item.name}" on page ${pageNumber} with filename: ${item.file}`);
                }
                found_automotive_assessment_item = true;
                /** extract options from the REPORTER Item name if they exist */
                let re = /options:\s*\{([^}]+)\}/gi;
                let options_str;
                let options_match = item.name.match(re);
                if (options_match) {
                    // extract the capture group
                    options_str = options_match[0];
                    f_contents.Write(`IMAGE,${item.file},${options_str}\n`);
                } else {
                    f_contents.Write(`IMAGE,${item.file}\n`);
                }
            }
        }
    }
    f_contents.Close();

    if (!found_automotive_assessment_item) {
        var v_job_control = Variable.GetFromName(templ, `JOB_CONTROL`);
        v_job_control.value = `Abort`;
        LogPrint(`Generation aborted because no automotive assessment items were found.`);
        Exit();
    }
}
