initialise_language_constant_reporter_variables();
summary_template();
/* Script for the CNCAP Summary Template scoring and generation */
/* autotableType is Reporter.AUTO_TABLE_DIRECTORY or Reporter.AUTO_TABLE_FILE valid on item type Item.AUTO_TABLE */

function summary_template() {
    let template = Template.GetCurrent();
    let page = template.GetPage(0);
    let star = ""; // Star to be displayed if something is missing

    let language = "English";

    /** if the template filename contains "_CN.or" then set the language to Chinese */
    if (/_CN\.or/i.test(template.filename)) {
        language = "Chinese";
    }

    let message_txt =
        "Select a directory to recursively search in to find the reporter_variables files for the following templates:\n" +
        "1x C-NCAP Far Side Protocol 2024 Working Conditions 1-8 with Correction Factor A\n" +
        "7x C-NCAP Far Side Protocol 2024 Working Conditions 1-8\n" +
        "1x C-NCAP Far Side Protocol 2024 Dual-Occupant Penalty (Passenger)\n" +
        "1x C-NCAP Far Side Protocol 2024 Dual-Occupant Penalty (Driver - only if Airbag is present)\n" +
        "The reporter_variables files will have been output to the selected output directories after the above templates have been run";

    if (language == "Chinese") {
        message_txt =
            "对单个模型运行与其工况相关的 REPORTER 报告模板之后，模型所在的文件夹内都会产生对应的变量文件 (reporter_variables) 用于存储该工况模型的结果与评分。当您运行完以下报告模板之后，即可使用本模板计算总体得分：\n" +
            "1x C-NCAP 2024 侧面远端虚拟测评 修正系数 A (抽检滑台工况 1 - 8)\n" +
            "7x C-NCAP 2024 侧面远端虚拟测评 滑台工况 1 - 8\n" +
            "1x C-NCAP 2024 侧面远端规程 双乘员罚分 (侧面柱碰, 副驾)\n" +
            "1x C-NCAP 2024 侧面远端规程 气囊对称性验证罚分 (副驾侧柱碰, 主驾) - 仅限配备远端乘员保护气囊的车型参与\n" +
            "请选择一个包含上述所有报告模板运行后产生的变量文件 (reporter_variables) 的文件夹，本模板会自动搜寻，并计算总体得分。";

        message_txt = get_unicode_text_with_padding(message_txt);
    }

    if (!Batch()) {
        let message = Window.Message("reporter_variables", message_txt, Window.OK | Window.CANCEL);

        if (message == Window.OK) {
            let dir = Window.GetDirectory();
            if (dir) {
                new Variable(
                    template,
                    "REPORTER_VARIABLES_DIR",
                    "Directory of reporter variable file",
                    dir,
                    "General",
                    false,
                    true
                );
            } else {
                LogError("No directory selected");
                return;
            }

            let output_message = Window.Message(
                "Output",
                "Select a directory to save the output summary template (CSV and XLSX) file\n" +
                    "Selecting 'No' will use the directory of the reporter_variables files",
                Window.OK | Window.NO
            );
            if (output_message == Window.OK) {
                let output_dir = Window.GetDirectory();
                if (output_dir) {
                    new Variable(template, "OUTPUT_DIR", "Output directory", output_dir, "General", false, true);
                } else {
                    new Variable(template, "OUTPUT_DIR", "Output directory", dir, "General", false, true);
                }
            } else {
                new Variable(template, "OUTPUT_DIR", "Output directory", dir, "General", false, true);
            }
        } else {
            LogError("No directory selected for reporter variables");
            return;
        }
    }

    if (Batch()) {
        let reporter_variables_dir = Variable.GetFromName(template, "REPORTER_VARIABLES_DIR").value;
        let output_dir = Variable.GetFromName(template, "OUTPUT_DIR").value;
        if (!File.IsDirectory(reporter_variables_dir) || !File.IsDirectory(output_dir)) {
            LogError(
                "REPORTER_VARIABLES_DIR or OUTPUT_DIR inputs from running in batch are not directories. Please check the inputs and try again"
            );
            return;
        }
    }

    let summary_item = Item.GetFromName(page, "Summary Table");
    summary_item.Generate();

    /* Loop through the generated auto table and find out what's what. There are 8 working conditions */
    let sim_scores = [];
    let penalty = 0;
    let working_conditions = 8;
    for (let i = 0; i < working_conditions; i++) {
        try {
            /* Get the test_name, current_score and cfa_string  */
            var test_name = summary_item.GetGeneratedData(i, 0);
            var current_score = Number(summary_item.GetGeneratedData(i, 7));
            var cfa_string = summary_item.GetGeneratedData(i, 8);

            /* cfa_string is set to "" automatically, so it if it contains a value then it is the CFA test, so assign cfa_string to cfa and test_score */
            if (cfa_string != "" && (test_name.includes("Working Condition") || test_name.includes("滑台工况"))) {
                var cfa = Number(cfa_string);
                var test_score = current_score;
            }

            /* If it is not a CFA test then it is a sim test, so push the current_score onto the sim_scores array */
            if (cfa_string == "" && (test_name.includes("Working Condition") || test_name.includes("滑台工况"))) {
                sim_scores.push(current_score);
            }

            /** If it is a penalty, then it is not a working condition test so we need to add one onto the loop to find the next working condition
             * If it has dropped in here then we need to add another loop to the working conditions */
            if (test_name.includes("Penalty") || test_name.includes("罚分")) {
                penalty += current_score;
                working_conditions++;
            }
        } catch (error) {
            LogError(
                "Could not read generated data from auto table. " +
                    "There should be a minimum of 8 rows (1x Test CFA, 7x Sim). " +
                    "The error occured on row " +
                    i +
                    ". Score for this row will be set to 0. The overall score may now not be accurate."
            );
            sim_scores.push(0);
            star = "*";
        }
    }

    /**
     * To calculate the total_score
     * First loop through the sim_scores and multiply each one by the cfa value
     * Then add the test score onto the total_score
     * Then divide by the number of working condition tests which is 8 (7 sim tests and 1 cfa test)
     * Then add the penalty
     */
    if (typeof cfa === "undefined") {
        LogError("CFA and Test Score not found, total score can not be calculated");
        return;
    }

    let total_score = 0;
    for (let i = 0; i < sim_scores.length; i++) {
        total_score += sim_scores[i] * cfa;
    }
    LogPrint("Total score after 7x simulations adjusted with CFA Value: " + total_score);
    total_score += test_score;
    LogPrint("Total score after test added: " + total_score);
    total_score /= 8;
    LogPrint("Total score after division: " + total_score);
    total_score += penalty;
    LogPrint("Total score after penalty applied: " + total_score);

    new Variable(template, "TOTAL_SCORE", "Total score for CNCAP", total_score.toPrecision(3), "General", false, true);

    /* Star to be displayed if something is missing */
    new Variable(template, "SUMMARY_ERROR", "Error", star, "General", false, true);
}

/**
 * There is a bug with REPORTER 21.1 where some unicode characters are not displayed correctly in the text box.
 * This function adds padding to the end of the text to ensure that the unicode characters are displayed correctly
 * @param {String} [text="😀"] - The text to be displayed in the text box
 * @param {Boolean} [debug=false] - If true, the function will print the text with padding and the variable value
 * @returns {String} text_with_padding
 */
function get_unicode_text_with_padding(text = "😀", debug = false) {
    let templ = Template.GetCurrent();
    /** create a temporary variable to set the value of */
    let rep_var = new Variable(templ, "UNICODE_TEXT_WITH_PADDING");
    /** set the value to the string plus padding of $ characters for 4 times the length */
    rep_var.value = text + "$".repeat(text.length * 4);
    /** get the value after setting and subtract the original length to see how many
     *  $ characters are "swallowed up" in the unicode conversion */
    let num_trailing_stars = rep_var.value.length - text.length;
    let text_with_padding = text + "$".repeat(text.length * 4 - num_trailing_stars);
    /** if debugging set the REPORTER variable value to confirm that it worked
     * otherwise remove the variable*/
    if (debug) {
        rep_var.value = text_with_padding;
        /** log print the string with padding and the variable value */
        LogPrint(text_with_padding);
        /** note that the variable value will not have the padding so it will need to have the padding
         * added again using get_unicode_text_with_padding() */
        LogPrint(rep_var.value);
        LogPrint(get_unicode_text_with_padding(rep_var.value, false));
    } else {
        rep_var.Remove();
    }
    return text_with_padding;
}

/**
 * This function initialises the REPORTER variables that are used in the script.
 * @param {Boolean} [override = false] - If true, the function will overwrite existing variables
 * otherwise it will only create them if they don't exist
 */
function initialise_language_constant_reporter_variables(override = false) {
    let templ = Template.GetCurrent();

    let language_constants = get_language_constants();

    let language = "English";

    /** if the template filename contains "_CN.or" then set the language to Chinese */
    if (/_CN\.or/i.test(templ.filename)) {
        language = "Chinese";
    }

    for (let const_name in language_constants) {
        /** note that language constants are wrapped in [] so that they are all grouped together in the variable list
         * and to reduce the risk of unintentionally overwriting a variable with the same name
         */
        let var_name = `[${const_name}]`;
        /** only create the language constant variables if they don't exist yet or if override is true */
        if (get_expanded_variable_value(templ, var_name) == null || override) {
            let temp_var = new Variable(
                templ,
                var_name,
                `${language} Language constant. The value may be changed to support a different language`,
                get_unicode_text_with_padding(language_constants[const_name][language]),
                "String",
                false,
                false // make language constant variables permanent
            );
        }
    }
}

/**
 * Returns an object containing the language constants used in the template
 * Add support for new languages and constants here
 * @returns {Object} -
 */
function get_language_constants() {
    /** note that the case matters */
    return {
        PASS: { English: "Pass", Chinese: "通过" },
        FAIL: { English: "Fail", Chinese: "未通过" },
        PASS_ALL_CAPS: { English: "PASS", Chinese: "通过" },
        FAIL_ALL_CAPS: { English: "FAIL", Chinese: "未通过" },
        MISSING: { English: "Missing", Chinese: "缺失" },
        MISSING_UPPER: { English: "MISSING", Chinese: "缺失" },
        YES: { English: "Yes", Chinese: "是" },
        NO: { English: "No", Chinese: "否" },
        NO_UPPER: { English: "NO", Chinese: "否" },
        YES_UPPER: { English: "YES", Chinese: "是" },
        NA: { English: "N/A", Chinese: "无" },
        CAPPING: { English: "CAPPING", Chinese: "上限区" },
        CAPPING_LIMIT_EXCEEDED: { English: "Capping limit exceeded", Chinese: "超出极限值" },
        STAR_CAPPING_LIMIT_EXCEEDED: { English: "*Capping limit exceeded", Chinese: "*超出极限值" },
        RED: { English: "RED", Chinese: "红色区" },
        BROWN: { English: "BROWN", Chinese: "棕色区" },
        CNCAP_RED: { English: "RED**", Chinese: "红色区**" },
        ORANGE: { English: "ORANGE", Chinese: "橙色区" },
        YELLOW: { English: "YELLOW", Chinese: "黄色区" },
        GREEN: { English: "GREEN", Chinese: "绿色区" },
        //Note that the following language constants are used in the summary template so you may
        //need to update the places they are used in cncap_summary.js if you make any changes below
        WORKING_CONDITION: { English: "Working Condition", Chinese: "滑台工况" },
        DRIVER_AIRBAG_PENALTY: { English: "Driver (Airbag) Penalty", Chinese: "主驾（气囊）罚分" },
        PASSENGER_PENALTY: { English: "Passenger Penalty", Chinese: "乘员罚分" },
        DESIGN_LOCATION: { English: "Design Location", Chinese: "设计位置" },
        HIGHEST_LEVEL: { English: "Highest Level", Chinese: "调至最高" },
        POLE_IMPACT: { English: "pole impact", Chinese: "柱碰" }
    };
}

/**
 * Wrapper function to return the expanded variable value, handling the case where it doesn't exist
 * @param {Template} templ The REPORTER template object
 * @param {string} name Variable name you want to get value for
 * @returns {?string}
 */
function get_expanded_variable_value(templ, name) {
    let value = templ.GetVariableValue(name);
    if (value == null) {
        return null;
    } else {
        let expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));

        /* recursively expand variables until no more variables are found */
        while (expanded_value != value) {
            LogPrint(`Expanded variable ${name} from "${value}" -> "${expanded_value}"`);
            value = expanded_value;
            expanded_value = templ.ExpandVariablesInString(get_unicode_text_with_padding(value));
        }

        /** there is a bug where the templ.ExpandVariablesInString() does not handle slashes correctly
         * and accidentally puts double slashes in the path.
         * This is not generally a problem except when working with UNC file paths
         * The workaround is to check if it starts with 4 slashes then remove 2 of them
         */
        if (/^[\/]{4}/.test(expanded_value)) {
            LogPrint(`Removing 2 slashes from the expanded variable value: ${expanded_value}`);
            expanded_value = expanded_value.substring(2); // Remove first two slashes
        }

        return expanded_value;
    }
}
